package com.atlassian.user;

import com.atlassian.user.search.page.Pager;
import com.atlassian.user.security.password.PasswordEncryptor;

public interface UserManager extends EntityManager
{
    /**
     * @return a {@link Pager} holding all users being managed.
     * @throws EntityException
     */
    Pager getUsers()  throws EntityException;

    /**
     * @return a {@link Pager} holding the names of all users being managed.
     * @throws EntityException
     */
    Pager getUserNames() throws EntityException;

    /**
     * @return a <code>null</code> or a {@link User} if one could be found.
     */
    User getUser(String username) throws EntityException;

    /**
     * @return a {@link User} object.
     *
     * @throws UnsupportedOperationException - if {@link UserManager#isCreative()} returns <code>false</code>.
     */
    User createUser(String username) throws EntityException;


    /**
     * Encrypts the plain password, sets it on the user, and saves the user.
     * Implementations supporting this will usually have an internal {@link PasswordEncryptor}.
     *
     * @throws UnsupportedOperationException - if {@link UserManager#isCreative()} returns <code>false</code>.
     */
    void alterPassword(User user, String plainTextPass) throws EntityException;

    /**
     * @throws UnsupportedOperationException - if {@link UserManager#isCreative()} returns <code>false</code>.
     */
    void saveUser(User user) throws EntityException;

    /**
     * @throws UnsupportedOperationException if {@link UserManager#isReadOnly(User)} returns <code>true</code>.
     * @throws IllegalArgumentException if the user is null or cannot be found in this repository
     */
    void removeUser(User user) throws EntityException, IllegalArgumentException;

    /**
     * @return true indicates that information on the user object cannot be altered in the storage system
     * (see {@link com.atlassian.user.repository.RepositoryIdentifier}),
     * false indicates that the storage system will save changes or that this {@link UserManager} does not 
     * know about the {@link User}.
     */
    boolean isReadOnly(User user) throws EntityException;
}

