package com.atlassian.user.configuration.xml;

import com.atlassian.user.util.ClassLoaderUtils;
import com.atlassian.user.util.FileUtils;
import com.atlassian.user.configuration.xml.XMLConfigurationParser;
import com.atlassian.user.configuration.*;
import org.apache.log4j.Logger;

import java.io.*;
import java.util.List;
import java.util.Iterator;

public class XMLConfiguration implements Configuration
{
    private static final Logger log = Logger.getLogger(XMLConfiguration.class);

    /**
     * the key of the system property containing the path to the atlassian user xml configuration file.
     */
    private static final String SYSTEM_ATLASSIAN_USER_CONFIGURATION = "atlassian.user.configuration";

    protected String fileName = "atlassian-user.xml";
    protected XMLConfigurationParser xmlParser;
    protected boolean initialized;
    protected boolean initializing;
    protected DelegationAccessor delegationAccessor;
    protected List repositoryConfigs;
    protected InputStream xmlIS;

    public static XMLConfiguration configureFromXMLString(String xmlString) throws ConfigurationException
    {
        ByteArrayInputStream in = new ByteArrayInputStream(xmlString.getBytes());
        try
        {
            XMLConfiguration config = new XMLConfiguration(in);
            config.init();
            return config;
        }
        finally
        {
            FileUtils.shutdownStream(in);
        }
    }

    public XMLConfiguration() throws ConfigurationException
    {
        xmlParser = new XMLConfigurationParser();
        delegationAccessor = new DefaultDelegationAccessor();
    }

    public XMLConfiguration(String fileName) throws ConfigurationException
    {
        this.fileName = fileName;
        xmlParser = new XMLConfigurationParser();
        delegationAccessor = new DefaultDelegationAccessor();
    }

    public XMLConfiguration(InputStream is) throws ConfigurationException
    {
        xmlIS = is;
        xmlParser = new XMLConfigurationParser();
        delegationAccessor = new DefaultDelegationAccessor();
    }

    public DelegationAccessor getDelegationAccessor()
    {
        if (!initialized && !initializing)
        {
            try
            {
                init();
            }
            catch (ConfigurationException e)
            {
                throw new RuntimeException("Atlassian User failed to initialize: " + e.getMessage(), e);
            }
        }

        return delegationAccessor;
    }

    public void init() throws ConfigurationException
    {
        if (!initialized && !initializing)
        {
            initializing = true;

            if (xmlIS == null)
                xmlIS = getXmlConfigurationFileAsInputStream();

            xmlParser.parse(xmlIS);

            try
            {
                xmlIS.close();
            }
            catch (IOException e)
            {
                throw new ConfigurationException("Could not close inputstream on [" + fileName + "]: "
                        + e.getMessage(), e);
            }

            repositoryConfigs = xmlParser.getRepositoryConfigurations();
            for (Iterator configIter = repositoryConfigs.iterator(); configIter.hasNext(); )
            {
                RepositoryConfiguration repositoryConfiguration = (RepositoryConfiguration) configIter.next();
                RepositoryAccessor accessor = configureRepository(repositoryConfiguration);
                delegationAccessor.addRepositoryAccessor(accessor);
            }

            initialized = true;
        }

        initializing = false;
    }

    /**
     * <p>Gets the {@link java.io.InputStream} of the xml configuration file to use.</p>
     * <p>This methods strategy is:
     * <ul>
     * <li>Check for the {#SYSTEM_ATLASSIAN_USER_CONFIGURATION} system property, if it exists use configuration file denoted by the given path,</li>
     * <li>Check for an "atlassian-user.xml" file in the classpath and use it if found</li>
     * </ul>
     * Whenever there is a problem with one specific configuration, it tries the next option, until there is no other option.
     * </p>
     *
     * @return the {@link InputStream} to the configuration file, <code>null</code> if no suitable configuration file was found.
     */
    private InputStream getXmlConfigurationFileAsInputStream()
    {
        InputStream is = null;

        // check the system property first
        String configurationFilePath = System.getProperty(SYSTEM_ATLASSIAN_USER_CONFIGURATION);
        if (configurationFilePath != null)
        {
            final File configurationFile = new File(configurationFilePath);
            if (configurationFile.exists())
            {
                try
                {
                    is = new BufferedInputStream(new FileInputStream(configurationFile));
                    if (log.isInfoEnabled())
                    {
                        log.info("Using configuration file at: [" + configurationFile.getAbsolutePath() + "]");
                    }
                }
                catch (IOException e)
                {
                    is = null;
                    log.warn("Couldn't load file at: [" + configurationFile.getAbsolutePath() + "], falling back on classpath resource.", e);
                }
            }
            else if (log.isDebugEnabled())
            {
                log.debug("Couldn't find file at [" + configurationFile.getAbsolutePath() + "], falling back on classpath resource.");
            }
        }
        else if (log.isDebugEnabled())
        {
            log.debug("System property " + SYSTEM_ATLASSIAN_USER_CONFIGURATION + " not specified, using classpath resource.");
        }

        // if input stream is still null, look for resource in the classpath
        if (is == null)
        {
            is = ClassLoaderUtils.getResourceAsStream(fileName, this.getClass());
        }

        return is;
    }

    protected RepositoryAccessor configureRepository(RepositoryConfiguration repositoryConfiguration) throws ConfigurationException
    {
        return repositoryConfiguration.configure();
    }

    public boolean isInitialized()
    {
        return initialized;
    }
}
