package com.atlassian.user.impl.hibernate.properties;

import com.atlassian.user.*;
import com.atlassian.user.impl.hibernate.DefaultHibernateExternalEntity;
import com.atlassian.user.impl.hibernate.DefaultHibernateUser;
import com.atlassian.user.impl.hibernate.ExternalEntityDAO;
import com.atlassian.user.impl.hibernate.HibernateUserManager;
import com.atlassian.user.impl.hibernate.repository.HibernateRepository;
import com.atlassian.user.properties.PropertySetFactory;
import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.module.propertyset.PropertySetManager;

import java.util.HashMap;

public class HibernatePropertySetFactory implements PropertySetFactory
{
    public static final String HIBERNATE_PROPERTY_SET = "hibernate";
    public static final String EXTERNAL_ENTITY = "EXT";
    public static final String LOCAL_USER = "LOC";

    protected final UserManager userManager;
    protected final ExternalEntityDAO externalEntityDAO;
    protected final HibernateRepository repository;

    public HibernatePropertySetFactory(UserManager userManager, ExternalEntityDAO externalEntityDAO, HibernateRepository repository)
    {
        this.userManager = userManager;
        this.externalEntityDAO = externalEntityDAO;
        this.repository = repository;
    }

    public PropertySet getPropertySet(Entity entity) throws EntityException
    {
        return getPropertySet(entity.getName());
    }

    /**
     * Retrieve profile for User with given name.
     */
    protected PropertySet getPropertySet(String entityName) throws EntityException
    {
        PropertySet propertySet;

        HashMap args = new HashMap();

        User user = null;
        if (userManager instanceof HibernateUserManager)
        {
            //attain the local user, as we need to ensure that this user is handled by this database
            //so that we can use the 'id' as a reference to pass to get the user's propertyset
            //this is due to propertysets needing a numeric id
            user = userManager.getUser(entityName);
            if (user != null)
            {
                args.put("entityId", new Long(((DefaultHibernateUser) user).getId()));
                args.put("entityName", LOCAL_USER + "_" + user.getName());
                args.put("configurationProvider", repository.getHibernateConfigurationProvider());
            }
        }

        if (user == null)
        {
            //these objects will be built via the call to Hibernate
            ExternalEntity externalEntity = externalEntityDAO.getExternalEntity(entityName);

            if (externalEntity == null)
            {
                externalEntity = new DefaultHibernateExternalEntity();
                ((DefaultHibernateExternalEntity) externalEntity).setName(entityName);
                ((DefaultHibernateExternalEntity) externalEntity).setType(EXTERNAL_ENTITY);
                externalEntityDAO.saveExternalEntity(externalEntity);
            }

            args.put("entityId", new Long(externalEntity.getId()));
            args.put("entityName", externalEntity.getType() + "_" + externalEntity.getName());
            args.put("configurationProvider", repository.getHibernateConfigurationProvider());
        }

        propertySet = getPropertySet(args);

        return propertySet;
    }

    protected PropertySet getPropertySet(HashMap args)
    {
        return PropertySetManager.getInstance(HIBERNATE_PROPERTY_SET, args);
    }
}
