package com.atlassian.user.impl.ldap.search;

/**
 * Handles all searches for LDAP. The managers simply send this component instructions.
 */

import com.atlassian.user.EntityException;
import com.atlassian.user.User;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.ldap.LDAPEntity;
import com.atlassian.user.impl.ldap.properties.LdapSearchProperties;
import com.atlassian.user.impl.ldap.repository.LdapContextFactory;
import com.atlassian.user.util.LDAPUtils;
import com.atlassian.util.profiling.UtilTimerStack;
import net.sf.ldaptemplate.support.filter.EqualsFilter;
import net.sf.ldaptemplate.support.filter.Filter;
import org.apache.log4j.Logger;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.DirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

public class DefaultLDAPUserAdaptor implements LDAPUserAdaptor
{
    protected final Logger log = Logger.getLogger(this.getClass());

    private final LdapContextFactory repository;
    private final LdapSearchProperties searchProperties;
    private final LdapFilterFactory filterFactory;

    public DefaultLDAPUserAdaptor(LdapContextFactory repository, LdapSearchProperties searchProperties,
        LdapFilterFactory filterFactory)
    {
        this.filterFactory = filterFactory;
        this.searchProperties = searchProperties;
        this.repository = repository;
    }

    private String[] getDefaultAttributes()
    {
        String[] defaultAttributesToReturn = new String[4];
        defaultAttributesToReturn[0] = searchProperties.getUsernameAttribute();
        defaultAttributesToReturn[1] = searchProperties.getFirstnameAttribute();
        defaultAttributesToReturn[2] = searchProperties.getSurnameAttribute();
        defaultAttributesToReturn[3] = searchProperties.getEmailAttribute();

        return defaultAttributesToReturn;
    }

    public LDAPPagerInfo search(Filter searchFilter) throws RepositoryException
    {
        return search(searchFilter, getDefaultAttributes());
    }

    public LDAPPagerInfo search(Filter userFilter, String[] attributesToReturnFromSearch) throws RepositoryException
    {
        DirContext ctx = null;

        SearchControls ctls = LDAPUtils.createSearchControls(attributesToReturnFromSearch,
            searchProperties.isUserSearchScopeAllDepths(),
            searchProperties.getTimeLimitMillis());

        NamingEnumeration userSearchEnume;
        Filter filter = LDAPUtils.makeAndFilter(filterFactory.getUserSearchFilter(), userFilter);

        try
        {
            ctx = repository.getLDAPContext();

            if (UtilTimerStack.isActive())
                UtilTimerStack.push(this.getClass().getName() + "_search_JNDI_RAW_" + filter);

            if (log.isDebugEnabled())
                log.debug("DefaultLDAPUserAdapter.search:" + filter.encode());

            userSearchEnume = ctx.search(searchProperties.getBaseUserNamespace(), filter.encode(), ctls);

            return new LDAPPagerInfo(userSearchEnume, filter,
                searchProperties.getBaseUserNamespace(),
                searchProperties.isUserSearchScopeAllDepths(),
                attributesToReturnFromSearch, searchProperties.getTimeLimitMillis());
        }
        catch (NamingException e)
        {
            throw new RepositoryException(e);
        }
        finally
        {
            try
            {
                if (ctx != null) ctx.close();
            }
            catch (NamingException e)
            {
                log.warn("Exception closing context", e);
            }

            if (UtilTimerStack.isActive())
                UtilTimerStack.pop(this.getClass().getName() + "_search_JNDI_RAW_" + filter);
        }

    }

    public LDAPPagerInfo getUserAttributes(String username, String[] specifiedAttributes) throws RepositoryException
    {
        Filter searchFilter = new EqualsFilter(searchProperties.getUsernameAttribute(), username);
        return search(LDAPUtils.makeAndFilter(filterFactory.getUserSearchFilter(), searchFilter), specifiedAttributes);
    }

    public String getUserDN(User user) throws EntityException
    {
        if (user instanceof LDAPEntity)
        {
            LDAPEntity entity = (LDAPEntity) user;
            return entity.getDistinguishedName();
        }

        return getUserDN(user.getName());
    }

    public String getUserDN(String username) throws EntityException
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_getUserDN(" + username + ")");

        LDAPPagerInfo ldapPagerInfo = getUserAttributes(username, getDefaultAttributes());

        if (ldapPagerInfo.getNamingEnumeration().hasMoreElements())
        {
            SearchResult result = (SearchResult) ldapPagerInfo.getNamingEnumeration().nextElement();

            String userDN = result.getName();
            // strip any quotes which JNDI has put around the name
            if (userDN.startsWith("\"") && userDN.endsWith("\""))
            {
                userDN = userDN.substring(1, userDN.length() - 1);
            }
            //we need a DN. The result only returns the full DN if the result was in the base getGroupEntries context
            if (userDN.indexOf(searchProperties.getBaseUserNamespace()) == -1)
                userDN = userDN + "," + searchProperties.getBaseUserNamespace();

            try
            {
                ldapPagerInfo.getNamingEnumeration().close();
            }
            catch (NamingException e)
            {
                throw new EntityException(e);
            }

            if (UtilTimerStack.isActive())
                UtilTimerStack.pop(this.getClass().getName() + "_getUserDN(" + username + ")");

            return userDN;
        }

        throw new RepositoryException("Could not locate a DN for user [" + username + "]");
    }

    /**
     * Builds a LDAP getGroupEntries filter matching RFC-2254 by ANDing the getGroupEntries term.
     *
     * @param firstTerm - a <code>StringBuffer</code> which will become the final getGroupEntries filter
     * @param addedFilter - a String holding a getGroupEntries Term to be ANDed.
     * @return <code>String</code> representing the completed firstTerm
     */
    public StringBuffer addSearchTermToFilter(StringBuffer firstTerm, String addedFilter)
    {
        if (firstTerm != null)
            if (addedFilter.indexOf("(") == 0 && addedFilter.lastIndexOf(")") == addedFilter.length() - 1)
                firstTerm = new StringBuffer("(&" + firstTerm + addedFilter + ")");
            else
                firstTerm = new StringBuffer("(&" + firstTerm + "(" + addedFilter + "))");
        else
            firstTerm = new StringBuffer(addedFilter);

        return firstTerm;
    }
}