package com.atlassian.user.impl.ldap.security.authentication;

import com.atlassian.user.EntityException;
import com.atlassian.user.impl.ldap.properties.LdapConnectionProperties;
import com.atlassian.user.impl.ldap.properties.LdapSearchProperties;
import com.atlassian.user.impl.ldap.repository.LdapContextFactory;
import com.atlassian.user.impl.ldap.search.DefaultLDAPUserAdaptor;
import com.atlassian.user.impl.ldap.search.LDAPUserAdaptor;
import com.atlassian.user.impl.ldap.search.LdapFilterFactory;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.security.authentication.Authenticator;
import com.atlassian.util.profiling.UtilTimerStack;
import net.sf.ldaptemplate.support.filter.AndFilter;
import net.sf.ldaptemplate.support.filter.EqualsFilter;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import javax.naming.NamingException;
import javax.naming.AuthenticationException;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import java.util.Hashtable;

public class DefaultLDAPAuthenticator implements Authenticator
{
    private static final Logger log = Logger.getLogger(DefaultLDAPAuthenticator.class);
    private final LDAPUserAdaptor userAdaptor;
    private final LdapSearchProperties searchProperties;
    private final RepositoryIdentifier repositoryIdentifier;
    private final LdapConnectionProperties connectionProperties;
    private final LdapFilterFactory filterFactory;
    private final LdapContextFactory contextFactory;

    public DefaultLDAPAuthenticator(RepositoryIdentifier repositoryIdentifier, LdapContextFactory contextFactory,
        LdapSearchProperties searchProperties, LdapConnectionProperties connectionProperties,
        LdapFilterFactory filterFactory)
    {
        this.repositoryIdentifier = repositoryIdentifier;
        this.filterFactory = filterFactory;
        this.searchProperties = searchProperties;
        this.connectionProperties = connectionProperties;
        this.contextFactory = contextFactory;
        this.userAdaptor = new DefaultLDAPUserAdaptor(contextFactory, searchProperties, filterFactory);
    }

    /**
     * Entering blank password will always fail, regardless of whether the underlying LDAP allows anonymous user
     * connects.
     * <p/>
     * This code duplicates the logic in the LDAPCredentialsProvider of OSUUser:  authenticate(String name, String
     * password)
     * <p/>
     * Sets up a LDAP DirContext (that will contain the credentials the user has just entered), and tries to perform a
     * search with it as a means of testing whether the credentials passed in are correct.
     *
     * @return true if the username and password were successfully authenticated, false otherwise
     */
    public boolean authenticate(String username, String password) throws EntityException
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_authenticate__" + username);

        // USER-140
        if (StringUtils.isEmpty(password))
        {
            if (log.isDebugEnabled())
                log.debug("Cannot perform authentication on empty passwords.");

            return false;
        }

        String userDN;
        DirContext authCtx = null;

        try
        {
            userDN = userAdaptor.getUserDN(username);
        }
        catch (EntityException e)
        {
            log.error("Could not construct DN to authenticate user: " + username, e);
            return false;
        }
        try
        {
            Hashtable authEnv = contextFactory.getAuthenticationJndiEnvironment(userDN, password);
            authCtx = new InitialDirContext(authEnv);

            // we need to attempt a search with this context to see if the user has been authenticated (its not enough to just create a context!)
            SearchControls ctls = new SearchControls();
            ctls.setReturningAttributes(new String[]{searchProperties.getUsernameAttribute()});
            ctls.setSearchScope(SearchControls.SUBTREE_SCOPE);

            AndFilter filter = new AndFilter();
            filter.and(filterFactory.getUserSearchFilter());
            filter.and(new EqualsFilter(searchProperties.getUsernameAttribute(), username));

            if (log.isDebugEnabled())
            {
                log.debug("Doing initial search to complete authentication, username: '" + username + "', " +
                    "base: '" + searchProperties.getBaseUserNamespace() + "' filter: '" + filter.encode() + "'");
            }

            // we don't care about the results, only that this search works
            authCtx.search(searchProperties.getBaseUserNamespace(), filter.encode(), ctls);
        }
        catch (AuthenticationException e)
        {
            if (log.isDebugEnabled())
                log.debug("LDAP authentication failed, user: '" + username + "', constructed DN: '" + userDN + "'", e);
            return false;
        }
        catch (NamingException e) // will also catch AuthenticationException's
        {
            log.error("LDAP authentication error, user: '" + username + "', " +
                "constructed DN: '" + userDN + "', connectionProperties: " + connectionProperties, e);
            return false;
        }
        catch (Throwable t)
        {
            log.error("Error occurred in LDAP authentication for username: " + username, t);
            return false;
        }
        finally
        {
            try
            {
                if (authCtx != null) authCtx.close();
            }
            catch (Exception e)
            {
                log.warn("Exception closing LDAP connection, possible resource leak", e);
            }

            if (UtilTimerStack.isActive())
                UtilTimerStack.pop(this.getClass().getName() + "_authenticate__" + username);
        }

        return true;
    }

    /**
     * Gets the repository which the authenticator belongs to
     */
    public RepositoryIdentifier getRepository()
    {
        return repositoryIdentifier;
    }
}