package com.atlassian.user.impl.osuser;

import com.atlassian.user.*;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.EntityNameAlphaComparator;
import com.opensymphony.user.ImmutableException;
import com.opensymphony.user.provider.AccessProvider;

import java.util.*;

import org.apache.log4j.Logger;

/**
 * An adaptor class for {@link com.opensymphony.user.provider.AccessProvider} and some of the higher level operations of
 * {@link com.opensymphony.user.UserManager}
 * <p/>
 * The rule is to use the credentialsProvider and/or profileProvider (for propertySets) for most things. Store()
 * operations must be called on the entity itself.
 */
public class OSUGroupManager extends OSUEntityManager implements GroupManager
{
    protected final Logger log = Logger.getLogger(this.getClass());

    private final OSUAccessor osuserAccessor;
    private final AccessProvider accessProvider;

    public OSUGroupManager(RepositoryIdentifier repository, OSUAccessor accessor) throws EntityException
    {
        super(repository);
        this.osuserAccessor = accessor;
        this.accessProvider = accessor.getAccessProvider();
    }

    public Pager getGroups()
    {
        SortedSet atlassianGroups = getGroupsFromAccessProvider();
        return new DefaultPager(atlassianGroups);
    }

    private SortedSet getGroupsFromAccessProvider()
    {
        SortedSet atlassianGroups = new TreeSet(new EntityNameAlphaComparator());
        for (Iterator iterator = accessProvider.list().iterator(); iterator.hasNext();)
        {
            String groupName = (String) iterator.next();
            Group atlassianGroup = getGroup(groupName);

            if (atlassianGroup != null)
                atlassianGroups.add(atlassianGroup);
        }
        return atlassianGroups;
    }

    public Group createGroup(String groupName) throws EntityException
    {
        Group group = null;

        if (accessProvider.handles(groupName))
            throw new com.atlassian.user.impl.DuplicateEntityException(
                "Group named [" + groupName + "] already exists in accessProvider ["
                    + accessProvider.toString());

        /**
         * Whenever a load(entityName) is called on a UserProvider the result
         * is, in fact, a reconstructed object, as below.
         *
         * This should obviously be cached.
         */
        if (accessProvider.create(groupName))
            group = new OSUGroup(new com.opensymphony.user.Group(groupName, osuserAccessor));

        return group;
    }

    public void removeGroup(Group group) throws EntityException, IllegalArgumentException
    {
        if (group == null)
            throw new IllegalArgumentException("Group is null.");
        else if (!(group instanceof OSUGroup))
            throw new IllegalArgumentException("User is not a OSUGroup [" + group.getClass().getName());

        Group groupToRemove = getGroup(group.getName());

        List users = accessProvider.listUsersInGroup(groupToRemove.getName());
        users = new ArrayList(users);

        for (int i = 0; i < users.size(); i++)
        {
            String username = (String) users.get(i);
            accessProvider.removeFromGroup(username, groupToRemove.getName());
        }

        accessProvider.remove(group.getName());
    }

    public void addMembership(Group group, User user)
    {
        if (group == null || getGroup(group.getName()) == null)
            throw new IllegalArgumentException(
                "Cannot add membership for unknown group: [" + (group == null ? "null" : group.getName()) + "]");

        accessProvider.addToGroup(user.getName(), group.getName());
    }

    public boolean hasMembership(Group group, User user)
    {
        /**
         * if the group is not an OpenSymphony wrapper, this manager will not have membership information
         * for the user.
         */
        if (!(group instanceof OSUGroup))
            return false;

        return accessProvider.inGroup(user.getName(), group.getName());
    }

    public void removeMembership(Group group, User user)
    {
        if (group == null || getGroup(group.getName()) == null)
            throw new IllegalArgumentException("Can't remove membership for unknown group: [" +
                (group == null ? "null" : group.getName()) + "]");
        accessProvider.removeFromGroup(user.getName(), group.getName());
    }

    public boolean isReadOnly(Group group) throws EntityException
    {
        return !(accessProvider.handles(group.getName()));
    }

    public boolean supportsExternalMembership() throws EntityException
    {
        return false;
    }

    public boolean supportsExternalMembership(Group group, User user) throws EntityException
    {
        return (user instanceof OSUUser) && osuserAccessor.getCredentialsProvider().handles(user.getName());
    }

    public Pager getMemberNames(Group group) throws EntityException
    {
        SortedSet atlUsers = new TreeSet(new EntityNameAlphaComparator());

        if (!(group instanceof OSUGroup))
        {
            return new DefaultPager(atlUsers);
        }

        Collection memberNames = accessProvider.listUsersInGroup(group.getName());

        for (Iterator iterator = memberNames.iterator(); iterator.hasNext();)
        {
            Object memberName = iterator.next();
            if (memberName != null)
                atlUsers.add(memberName);
        }

        return new DefaultPager(atlUsers);
    }

    public Pager getLocalMemberNames(Group group) throws EntityException
    {
        SortedSet atlUsers = new TreeSet(new EntityNameAlphaComparator());

        if (!(group instanceof OSUGroup))
        {
            return new DefaultPager(atlUsers);
        }

        List memberNames = new ArrayList(accessProvider.listUsersInGroup(group.getName()));
        Collections.sort(memberNames, new EntityNameAlphaComparator());

        return new DefaultPager(memberNames);
    }

    public Pager getExternalMemberNames(Group group) throws EntityException
    {
        throw new UnsupportedOperationException("External membership is not supported.");
    }

    public void saveGroup(Group group) throws EntityException
    {
        if (!(accessProvider.handles(group.getName())))
            return;

        com.opensymphony.user.Group g = new com.opensymphony.user.Group(group.getName(), osuserAccessor);

        try
        {
            g.store();
        }
        catch (ImmutableException e)
        {
            throw new RepositoryException(e);
        }
    }

    public Group getGroup(String groupName)
    {
        if (!accessProvider.handles(groupName))
            return null;

        com.opensymphony.user.Group osgroup = new com.opensymphony.user.Group(groupName, osuserAccessor);
        return new OSUGroup(osgroup);
    }

    public Pager getGroups(User user) throws RepositoryException
    {
        SortedSet groupsForUser = new TreeSet(new EntityNameAlphaComparator());
        Collection groupNamesForUser;

        try
        {
            groupNamesForUser = accessProvider.listGroupsContainingUser(user.getName());

            for (Iterator iterator = groupNamesForUser.iterator(); iterator.hasNext();)
            {
                String groupName = (String) iterator.next();
                groupsForUser.add(getGroup(groupName));
            }
        }
        catch (NullPointerException e)
        {

            // HACK: we shouldn't be ignoring this, but I'll act defensively for now by just adding a WARN
            log.warn("Errant null pointer in OSUGroupManager", e);
        }

        return new DefaultPager(groupsForUser);
    }

    public List /*<Group>*/ getWritableGroups()
    {
        return new ArrayList(getGroupsFromAccessProvider());
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        return repository;
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        if (getGroup(entity.getName()) != null)
            return repository;

        return null;
    }

    /**
     * @return mutability is hard-coded within each {@link com.opensymphony.user.provider.UserProvider} then applied to
     *         the entity produced (here a {@link com.opensymphony.user.Group}).
     *         <p/>
     *         All providers in the base OSUUser package are hardcoded to be mutable. This implementation returns the
     *         mutability value of the first group returned by {@link com.opensymphony.user.provider.UserProvider#list()}.
     *         <p/>
     *         If there are no groups it returns true.
     *         <p/>
     *         Thus, if you want immutable groups you should override this method or ensure that there is at least one
     *         entity which is handled by the provider. All groups should have the same mutability value.
     */
    public boolean isCreative()
    {
        Collection groups = accessProvider.list();

        if (groups.isEmpty())
            return true;

        String groupName = (String) groups.toArray(new String[]{})[0];

        return new com.opensymphony.user.Group(groupName, osuserAccessor).isMutable();
    }
}