package com.atlassian.user.util.migration;

import com.atlassian.user.*;
import com.atlassian.user.configuration.DefaultDelegationAccessor;
import com.atlassian.user.configuration.DelegationAccessor;
import com.atlassian.user.configuration.RepositoryAccessor;
import com.atlassian.user.impl.DefaultUser;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.hibernate.DefaultHibernateUser;
import com.atlassian.user.impl.hibernate.properties.HibernatePropertySetFactory;
import com.atlassian.user.impl.osuser.OSUAccessor;
import com.atlassian.user.impl.osuser.OSUUserManager;
import com.opensymphony.user.provider.AccessProvider;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.Session;
import net.sf.hibernate.SessionFactory;
import org.apache.log4j.Logger;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.datasource.SingleConnectionDataSource;
import org.springframework.orm.hibernate.SessionFactoryUtils;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

/**
 * Makes a raw JDBC connection to os_user tables and copies across information into the supplied {@link
 * com.atlassian.user.UserManager}, {@link com.atlassian.user.GroupManager}, and {@link
 * com.atlassian.user.properties.PropertySetFactory}
 */

public class OSUEntityMigrator implements EntityMigrator
{
    private static final Logger log = Logger.getLogger(OSUEntityMigrator.class);

    private static final String OSUSER_REPOSITORY_KEY = "osuserRepository";

    private UserManager targetUserManager;
    private GroupManager targetGroupManager;

    private AccessProvider osAccessProvider;

    private final SessionFactory sessionFactory;

    public OSUEntityMigrator(RepositoryAccessor osuserRepositoryAccessor, RepositoryAccessor repositoryAccessor, SessionFactory sessionFactory)
    {
        if (osuserRepositoryAccessor == null)
            throw new IllegalArgumentException("osuserRepositoryAccessor is required.");
        if (repositoryAccessor == null)
            throw new IllegalArgumentException("targetRepositoryAccessor is required.");
        if (sessionFactory == null)
            throw new IllegalArgumentException("sessionFactory is required.");

        this.sessionFactory = sessionFactory;

        final DelegationAccessor targetRepositoryAccessor = getNonOSUserRepositoryAccessor(repositoryAccessor);
        if (!targetRepositoryAccessor.getRepositoryAccessors().isEmpty())
        {
            final UserManager osUserManager = osuserRepositoryAccessor.getUserManager();
            if (osUserManager == null)
                throw new IllegalArgumentException("osUserManager is required.");

            final OSUAccessor osuAccessor = ((OSUUserManager) osUserManager).getAccessor();
            if (osuAccessor == null)
                throw new IllegalArgumentException("osuAccessor is required.");

            osAccessProvider = osuAccessor.getAccessProvider();
            if (osAccessProvider == null)
                throw new IllegalArgumentException("osAccessProvider is required.");

            targetUserManager = targetRepositoryAccessor.getUserManager();
            targetGroupManager = targetRepositoryAccessor.getGroupManager();

            if (targetUserManager == null)
                throw new IllegalArgumentException("userManager is required.");
            if (targetGroupManager == null)
                throw new IllegalArgumentException("groupManager is required.");
        }
    }

    private DelegationAccessor getNonOSUserRepositoryAccessor(RepositoryAccessor repositoryAccessor)
    {
        final DelegationAccessor nonOSUserDelegationAccessor = new DefaultDelegationAccessor();
        if (repositoryAccessor instanceof DelegationAccessor)
        {
            final DelegationAccessor delegationAccessor = (DelegationAccessor) repositoryAccessor;
            for (Iterator iterator = delegationAccessor.getRepositoryAccessors().iterator(); iterator.hasNext();)
            {
                final RepositoryAccessor accessor = (RepositoryAccessor) iterator.next();
                if (!OSUSER_REPOSITORY_KEY.equals(accessor.getIdentifier().getKey()))
                    nonOSUserDelegationAccessor.addRepositoryAccessor(accessor);
            }
            return nonOSUserDelegationAccessor;
        }
        else
        {
            if (!OSUSER_REPOSITORY_KEY.equals(repositoryAccessor.getIdentifier().getKey()))
                nonOSUserDelegationAccessor.addRepositoryAccessor(repositoryAccessor);
        }
        return nonOSUserDelegationAccessor;
    }

    /**
     * The method is organised in a 'strange' way for performance reasons. DON'T change it. Every white space is there for a reason :)
     * The performace problem was: every time we add a member to a hibernate group, hibernate marks this group object as dirty.
     * Because hibernate group contains the list of all its members and needs to iterate through all of them when flush() is called.
     * flush() is called every time hibernate thinks it needs to do it, for example when we call getUser() it first calls flush to make
     * sure we get up to date data. We structured the code in a way so that flush is not called until we add all users to a group.
     * That's why we cache the list of users. We also rely on the fact that targetGroupManager maintains its own cache and does not need
     * to call flush() each time we get a group by a groupname.
     *
     * @throws RepositoryException if there is no non OSUser repository was included in target repository Accessor
     */
    public void migrate(MigratorConfiguration config, MigrationProgressListener progressListener) throws EntityException
    {
        if (targetUserManager == null)
        {
            throw new RepositoryException("No non OSUser repository configured. Cannot perform migration.");
        }

        OSUserDao osUserDao = getOSUserDao();
        final Map users = osUserDao.findAllUsers();
        final Map userGroups = osUserDao.findAllUserGroups(users);
        Map migratedUsers = migrateUsers(progressListener, users);

        // migrate group memberships
        for (Iterator it = migratedUsers.entrySet().iterator(); it.hasNext();)
        {
            Map.Entry userEntry = (Map.Entry) it.next();
            final User user = (User) userEntry.getKey();
            migrateUserGroupMembership(user, (List) userGroups.get(user.getName()), ((Boolean) userEntry.getValue()).booleanValue(), config, progressListener);
        }

        // some groups could be empty hence they would not be migrated
        // when migrating users' membership -- need to migrate them explicitly
        migrateGroups(progressListener);
    }

    private Map migrateUsers(MigrationProgressListener progressListener, Map users)
            throws EntityException
    {
        // starting user migration
        progressListener.userMigrationStarted(users.size());

        Map migratedUsers = new HashMap();

        int i = 0;
        for (Iterator it = users.entrySet().iterator(); it.hasNext(); i++)
        {
            Map.Entry userEntry = (Map.Entry) it.next();
            final Long osUserId = (Long) userEntry.getKey();
            final User user = (User) userEntry.getValue();

            User existingUser = targetUserManager.getUser(user.getName());
            if(existingUser == null)
            {
                User newUser = addUser(targetUserManager, (DefaultUser) user);
                migratedUsers.put(newUser, new Boolean(true));
            }
            else
            {
                migratedUsers.put(existingUser, new Boolean(false));
            }
            migratePropertySet(osUserId, user);


            progressListener.userMigrated();
            if (i % 100 == 0){
                Session session = SessionFactoryUtils.getSession(sessionFactory, false);
                try
                {
                    session.flush();
                    session.clear();
                }
                catch (HibernateException e)
                {
                    log.error(e);
                }
            }
        }
        // users migrated, starting group migration
        progressListener.userMigrationComplete();
        return migratedUsers;
    }

    private OSUserDao getOSUserDao()
    {
        // we cannot cache dataSource because it is bound to a particular session
        final DataSource dataSource = getDataSource();
        return new OSUserDao(dataSource);
    }
    
    private void migrateGroups(MigrationProgressListener progressListener)
            throws EntityException
    {
        final List groups = osAccessProvider.list();
        progressListener.groupMigrationStarted(groups.size());

        for (Iterator groupsIterator = groups.iterator(); groupsIterator.hasNext();)
        {
            final String groupName = (String) groupsIterator.next();
            getOrCreateGroup(groupName);
            progressListener.groupMigrated();
        }
        // group migration complete
        progressListener.groupMigrationComplete();
    }

    private void migrateUserGroupMembership(User user, List userGroups, boolean isCreatedUser, MigratorConfiguration config,
                                            MigrationProgressListener progressListener) throws EntityException
    {
        if (userGroups != null){
            for(Iterator iter = userGroups.iterator();  iter.hasNext();){
                String groupName = (String) iter.next();
                Group group = getOrCreateGroup(groupName);
                if (isCreatedUser || config.isMigrateMembershipsForExistingUsers())
                {
                    if (log.isInfoEnabled()) log.info("Adding member <" + user.getName() + "> to group <" + groupName + ">");
                    if (!targetGroupManager.isReadOnly(group))
                    {
                        targetGroupManager.addMembership(group, user);
                    }
                    else
                    {
                        progressListener.readonlyGroupMembershipNotMigrated(group.getName(), user.getName());
                    }
                }
            }
        }
    }

    private void migratePropertySet(Long userId, User user) throws EntityException
    {
        if (log.isInfoEnabled()) log.info("Migrating properties for <" + user.getName() + ">");

        final User targetUser = targetUserManager.getUser(user.getName());
        final String entityName = getEntityName(targetUser);
        final long entityId = getEntityId(targetUser);

        final JdbcTemplate template = new JdbcTemplate(getDataSource());

        if (template.queryForInt("SELECT count(*) FROM OS_PROPERTYENTRY WHERE ENTITY_NAME=? AND ENTITY_ID=?", new Object[] {entityName, new Long(entityId)}) == 0)
        {
            template.query("SELECT * FROM OS_PROPERTYENTRY WHERE ENTITY_NAME = 'OSUser_user' AND ENTITY_ID = ? AND ENTITY_KEY <> 'fullName' AND ENTITY_KEY <> 'email'", new Object[]{userId}, new RowCallbackHandler()
            {
                public void processRow(ResultSet resultSet) throws SQLException
                {
                    template.update("INSERT INTO OS_PROPERTYENTRY (ENTITY_NAME,ENTITY_ID,ENTITY_KEY,KEY_TYPE,BOOLEAN_VAL,DOUBLE_VAL,STRING_VAL,LONG_VAL,INT_VAL,DATE_VAL) VALUES (?,?,?,?,?,?,?,?,?,?)", new Object[]{
                        entityName,
                        new Long(entityId),
                        resultSet.getString("ENTITY_KEY"),
                        new Integer(resultSet.getInt("KEY_TYPE")),
                        Boolean.valueOf(resultSet.getBoolean("BOOLEAN_VAL")),
                        new Double(resultSet.getDouble("DOUBLE_VAL")),
                        resultSet.getString("STRING_VAL"),
                        new Long(resultSet.getLong("LONG_VAL")),
                        new Integer(resultSet.getInt("INT_VAL")),
                        resultSet.getTimestamp("DATE_VAL")
                    });
                }
            });
        }
    }

    private String getEntityName(User user) throws EntityException
    {
        if (user instanceof DefaultHibernateUser)
        {
            return HibernatePropertySetFactory.LOCAL_USER + "_" + user.getName();
        }
        else if (user instanceof ExternalEntity)
        {
            return HibernatePropertySetFactory.EXTERNAL_ENTITY + "_" + user.getName();
        }
        else
        {
            throw new EntityException("Could not determine entityName for user: " + user + " of type: " + (user != null ? user.getClass().getName() : "unkown"));
        }
    }

    private long getEntityId(User user) throws EntityException
    {
        if (user instanceof DefaultHibernateUser)
        {
            return ((DefaultHibernateUser) user).getId();
        }
        else if (user instanceof ExternalEntity)
        {
            return ((ExternalEntity) user).getId();
        }
        else
        {
            throw new EntityException("Could not find id for user: " + user + " of type: " + (user != null ? user.getClass().getName() : "unkown"));
        }
    }

    /**
     * Adds the given user using the user manager.
     *
     * @param userManager the user manager to add the user to
     * @param user the user to add
     * @throws EntityException if a problem occur adding the user in the user manager
     */
    private User addUser(UserManager userManager, DefaultUser user) throws EntityException
    {
        if (log.isInfoEnabled()) log.info("Adding user <" + user.getName() + ">");

        final User newUser = userManager.createUser(user.getName());
        newUser.setFullName(user.getFullName());
        newUser.setEmail(user.getEmail());
        newUser.setPassword(user.getPassword());
        userManager.saveUser(newUser);
        return newUser;
    }

    private Group getOrCreateGroup(String groupName) throws EntityException
    {
        Group group = targetGroupManager.getGroup(groupName);
        if (group == null)
        {
            if (log.isInfoEnabled()) log.info("Creating group <" + groupName + ">");
            group = targetGroupManager.createGroup(groupName);
        }

        return group;
    }

    private DataSource getDataSource()
    {
        Session hibernateSession = SessionFactoryUtils.getSession(sessionFactory, true);
        Connection conn;
        try
        {
            conn = hibernateSession.connection();
        }
        catch (HibernateException e)
        {
            throw new RuntimeException(e);
        }
        return new SingleConnectionDataSource(conn, true);
    }

}
