package com.vaadin.copilot.customcomponent;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.vaadin.copilot.CopilotJacksonUtils;
import com.vaadin.copilot.FlowUtil;
import com.vaadin.copilot.JavaReflectionUtil;
import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.copilot.RouteHandler;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.ComponentInfoFinder;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.FlowComponentQuirks;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.copilot.javarewriter.JavaFileSourceProvider;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.router.RouteBaseData;
import com.vaadin.flow.router.RouteData;
import com.vaadin.flow.server.VaadinSession;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.JsonNodeType;

import com.github.javaparser.ast.expr.MethodCallExpr;

/**
 * Helper class for custom components
 */
public class CustomComponentHelper {
    private static final String ACTIVE_CLASS_JSON_KEY = "activeClass";
    private static final String ACTIVE_NODE_ID_JSON_KEY = "activeNodeId";

    private static final String JAVA_CLASS_NAME_REQ_KEY = "javaClassName";

    private CustomComponentHelper() {

    }

    /**
     * Generates the response that is used in the UI
     *
     * @param allComponents
     *            Component and source location of all components that are found in
     *            the UI
     * @param componentsInProject
     *            Components that have source in the project.
     * @return response that contains required metadata for the custom component
     *         feature
     */
    public static CustomComponentResponseData generateResponse(VaadinSession vaadinSession,
            Map<Component, ComponentTypeAndSourceLocation> allComponents, List<Component> componentsInProject) {

        Map<Integer, ComponentInfoForCustomComponentSupport> allComponentsInfoForCustomComponentSupport = new HashMap<>();
        CustomComponentResponseData responseData = new CustomComponentResponseData(
                allComponentsInfoForCustomComponentSupport);
        componentsInProject.stream().filter(component -> CustomComponents.isCustomComponent(component.getClass()))
                .forEach(component -> {
                    Integer nodeId = FlowUtil.getNodeId(component);

                    CustomComponent customComponent = CustomComponents.getCustomComponentInfo(component).orElseThrow();
                    String activeLevel;
                    if (CustomComponent.Type.IN_PROJECT.equals(customComponent.getType())) {
                        activeLevel = ((CustomComponentInProject) customComponent).sourceFile().getName();
                    } else {
                        activeLevel = "External [" + customComponent.componentClass().getSimpleName() + "]";
                    }
                    String filePath = null;
                    if (customComponent instanceof CustomComponentInProject customComponentInProject) {
                        filePath = customComponentInProject.sourceFile().getPath();
                    }
                    CustomComponentInstanceInfo customComponentInstanceInfo = new CustomComponentInstanceInfo(
                            customComponent.getType(), activeLevel, customComponent.componentClass().getName(),
                            filePath, customComponent.litTemplate());
                    allComponentsInfoForCustomComponentSupport.put(nodeId, customComponentInstanceInfo);
                });

        componentsInProject.forEach(component -> {
            ComponentTypeAndSourceLocation typeAndSourceLocation = allComponents.get(component);
            Integer nodeId = FlowUtil.getNodeId(component);
            Optional<ComponentTracker.Location> locationInProject = typeAndSourceLocation.createLocationInProject();
            String filePath = null;
            if (locationInProject.isPresent()) {
                filePath = ProjectFileManager.get().getSourceFile(locationInProject.get()).getPath();
            }
            if (!allComponentsInfoForCustomComponentSupport.containsKey(nodeId)) {
                allComponentsInfoForCustomComponentSupport.put(nodeId, new ComponentInfoForCustomComponentSupport());
            }
            boolean childOfCustomComponent = FlowUtil.testAncestors(vaadinSession, component,
                    ancestor -> CustomComponents.isCustomComponent(ancestor.getClass()));
            allComponentsInfoForCustomComponentSupport.get(nodeId).setCreateLocationPath(filePath);
            allComponentsInfoForCustomComponentSupport.get(nodeId).setChildOfCustomComponent(childOfCustomComponent);
            locationInProject.ifPresent(location -> allComponentsInfoForCustomComponentSupport.get(nodeId)
                    .setCreatedClassName(location.className()));
        });

        Integer routeNodeId = getRouteNodeId(componentsInProject, vaadinSession);
        if (allComponentsInfoForCustomComponentSupport.containsKey(routeNodeId)) {
            allComponentsInfoForCustomComponentSupport.get(routeNodeId).setRouteView(true);
        }
        return responseData;
    }

    private static Integer getRouteNodeId(Collection<Component> components, VaadinSession vaadinSession) {
        List<RouteData> serverRoutes = RouteHandler.getServerRoutes(vaadinSession);
        if (serverRoutes == null) {
            return null;
        }
        List<? extends Class<? extends Component>> navigationTargets = serverRoutes.stream()
                .map(RouteBaseData::getNavigationTarget).toList();
        Optional<Component> first = components.stream().filter(comp -> navigationTargets.contains(comp.getClass()))
                .findFirst();
        return first.map(FlowUtil::getNodeId).orElse(null);

    }

    /**
     * Checks if the given component is drilled down
     *
     * @param vaadinSession
     *            Vaadin session to access component
     * @param data
     *            Sent data from client. It requires
     *            {@link CustomComponentHelper#ACTIVE_CLASS_JSON_KEY} and
     *            {@link CustomComponentHelper#ACTIVE_NODE_ID_JSON_KEY}
     * @param component
     *            JSON data that contains uiId and nodeId
     * @return true if drilled down component, false otherwise
     */
    public static boolean isDrilledDownComponent(VaadinSession vaadinSession, JsonNode data, JsonNode component) {
        if (component == null) {
            return false;
        }
        return isDrilledDownComponent(vaadinSession, data, component.get("uiId").asInt(),
                component.get("nodeId").asInt());
    }

    /**
     * Checks if the given component is drilled down.
     *
     * @param vaadinSession
     *            Vaadin session to access component
     * @param data
     *            Sent data from client. It requires
     *            {@link CustomComponentHelper#ACTIVE_CLASS_JSON_KEY} and
     *            {@link CustomComponentHelper#ACTIVE_NODE_ID_JSON_KEY}
     * @param uiId
     *            uiId
     * @param nodeId
     *            nodeId of the given component
     * @return true if drilled down component, false otherwise
     */
    public static boolean isDrilledDownComponent(VaadinSession vaadinSession, JsonNode data, int uiId, int nodeId) {
        if (!data.has(ACTIVE_CLASS_JSON_KEY) || data.get(ACTIVE_CLASS_JSON_KEY).isNull()) {
            return false;
        }
        if (!data.has(ACTIVE_NODE_ID_JSON_KEY) || data.get(ACTIVE_NODE_ID_JSON_KEY).isNull()) {
            return false;
        }
        String activeClass = data.get(ACTIVE_CLASS_JSON_KEY).asText();
        int activeNodeId = data.get(ACTIVE_NODE_ID_JSON_KEY).asInt();
        if (activeNodeId != nodeId) {
            return false;
        }

        Optional<Component> componentByNodeIdAndUiId = FlowUtil.findComponentByNodeIdAndUiId(vaadinSession, nodeId,
                uiId);
        if (componentByNodeIdAndUiId.isEmpty()) {
            return false;
        }
        if (!componentByNodeIdAndUiId.get().getClass().getName().equals(activeClass)) {
            return false;
        }
        return true;
    }

    /**
     * Returns the hierarchy of dropped element onto a custom component. Extracts
     * the relevant parts from the data object.
     *
     * @param data
     *            should contain either <code>javaClassName</code> or
     *            <code>reactTag</code> to find relevant Java Class
     * @return list of
     * @throws CopilotException
     *             when something goes wrong with looking for class hierarchy
     */
    public static List<String> getClassHierarchyOfJavaClassNameOrReactTagFromRequest(JsonNode data) {
        List<String> classHierarchy = new ArrayList<>();
        try {

            if (data.hasNonNull(JAVA_CLASS_NAME_REQ_KEY)) {
                classHierarchy = FlowComponentQuirks.getClassHierarchy(new JavaComponent(null,
                        data.get(JAVA_CLASS_NAME_REQ_KEY).asText(), new HashMap<>(), new ArrayList<>()));
            } else if (data.hasNonNull("reactTag")) {
                classHierarchy = FlowComponentQuirks.getClassHierarchy(
                        new JavaComponent(data.get("reactTag").asText(), null, new HashMap<>(), new ArrayList<>()));
            }
            return classHierarchy;

        } catch (Exception e) {
            throw new CopilotException("Unable to get class hierarchy", e);
        }
    }

    /**
     * Filters the method in the custom component based on the dragged element
     * hierarchy. e.g. when you drop an Image onto Custom Component, the listed
     * methods should be in the hierarchy of Image. Methods that gets Button or any
     * other component as parameter should be filtered out.
     *
     * @param draggedElementClassHierarchy
     *            Hierarchy of dragged component. Strongly related to
     *            {@link #getClassHierarchyOfJavaClassNameOrReactTagFromRequest(JsonNode)}
     * @param methods
     *            List of methods that accept {@link Component} as parameter.
     * @return List of filtered components
     */
    public static List<JavaReflectionUtil.ComponentAddableMethod> filterMethodBasedOnDraggedComponentHierarchy(
            List<JavaReflectionUtil.ComponentAddableMethod> methods, List<String> draggedElementClassHierarchy) {
        return methods.stream().filter(
                childAddableMethod -> draggedElementClassHierarchy.contains(childAddableMethod.paramJavaClassName()))
                .toList();
    }

    /**
     * Updates add/replace flags based on its usage
     *
     * @param componentTypeAndSourceLocation
     *            Component type and source location
     * @param methods
     *            Methods belonging to the given component
     */
    public static void addInfoFromComponentSource(ComponentTypeAndSourceLocation componentTypeAndSourceLocation,
            List<CustomComponentAddMethodInfo> methods) throws IOException {
        ComponentInfoFinder finder = new ComponentInfoFinder(new JavaFileSourceProvider(),
                componentTypeAndSourceLocation);
        ComponentInfo componentInfo = finder.find();
        for (CustomComponentAddMethodInfo filteredMethod : methods) {
            String methodName = filteredMethod.getMethodName();
            List<MethodCallExpr> methodCalls = JavaRewriterUtil.findMethodCalls(componentInfo);
            for (MethodCallExpr methodCall : methodCalls) {
                if (methodCall.getNameAsString().equals(methodName)) {
                    filteredMethod.setReplace(true);
                }
            }
            List<List<JavaReflectionUtil.ParameterTypeInfo>> allMethodParameterTypes = JavaReflectionUtil
                    .getAllMethodParameterTypes(componentTypeAndSourceLocation.type(), methodName);
            boolean hasMethod = allMethodParameterTypes.stream().anyMatch(parameterTypes -> parameterTypes.size() == 1
                    && (parameterTypes.get(0).type().isArray() || parameterTypes.get(0).type().isVarArgs()));
            if (hasMethod) {
                filteredMethod.setAdd(true);
            }
            if (!filteredMethod.isAdd() && !filteredMethod.isReplace()) {
                filteredMethod.setAdd(true);
            }
        }
    }

    /**
     * Calls {@link #extractFromRequest(JsonNode, String)} with
     * <code>customComponentApiSelection</code> key
     *
     * @param data
     *            sent from the Client
     * @return Null if customComponentApiSelection is not present in the JSON or not
     *         an object or unable to parse
     */
    public static CustomComponentApiSelection extractFromRequest(JsonNode data) {
        return extractFromRequest(data, "customComponentApiSelection");
    }

    /**
     * Extract {@link CustomComponentApiSelection} from the data witht the given key
     * if present.
     *
     * @param data
     *            sent from the client
     * @param key
     *            JSON key to get the value
     * @return CustomComponentApiSelection if present, or null if
     *         customComponentApiSelection is not present in the JSON or not an
     *         object or unable to parse
     */
    public static CustomComponentApiSelection extractFromRequest(JsonNode data, String key) {
        if (!data.has(key)) {
            return null;
        }
        JsonNode jsonNode = data.get(key);
        if (jsonNode.getNodeType() != JsonNodeType.OBJECT) {
            return null;
        }
        return CopilotJacksonUtils.readValue(jsonNode.toString(), CustomComponentApiSelection.class);
    }
}
