package com.vaadin.copilot.javarewriter;

import static com.vaadin.copilot.ConnectToService.HILLA_FILTER_TYPE;
import static com.vaadin.copilot.ConnectToService.PAGEABLE_TYPE;
import static com.vaadin.copilot.ConnectToService.STRING_TYPE;
import static com.vaadin.copilot.javarewriter.DataEntityRecordRewriter.DEFAULT_ENTITY_RECORD_NAME;

import java.lang.reflect.Constructor;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.vaadin.copilot.JavaReflectionUtil;
import com.vaadin.copilot.UIServiceCreator;
import com.vaadin.copilot.Util;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.copilot.javarewriter.custom.CustomComponentHandle;
import com.vaadin.copilot.javarewriter.custom.CustomComponentHandler;
import com.vaadin.flow.component.Composite;
import com.vaadin.flow.shared.util.SharedUtil;

import com.github.javaparser.Range;
import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.AccessSpecifier;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.Modifier;
import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.body.Parameter;
import com.github.javaparser.ast.body.TypeDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.comments.BlockComment;
import com.github.javaparser.ast.comments.Comment;
import com.github.javaparser.ast.comments.LineComment;
import com.github.javaparser.ast.expr.ArrayCreationExpr;
import com.github.javaparser.ast.expr.ArrayInitializerExpr;
import com.github.javaparser.ast.expr.AssignExpr;
import com.github.javaparser.ast.expr.BooleanLiteralExpr;
import com.github.javaparser.ast.expr.ClassExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.FieldAccessExpr;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.MethodReferenceExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.github.javaparser.ast.expr.ThisExpr;
import com.github.javaparser.ast.expr.VariableDeclarationExpr;
import com.github.javaparser.ast.nodeTypes.NodeWithArguments;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExplicitConstructorInvocationStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.stmt.ReturnStmt;
import com.github.javaparser.ast.stmt.Statement;
import com.github.javaparser.ast.type.ClassOrInterfaceType;
import com.github.javaparser.ast.type.Type;
import com.github.javaparser.resolution.Resolvable;
import com.github.javaparser.resolution.UnsolvedSymbolException;
import com.github.javaparser.resolution.declarations.ResolvedDeclaration;
import com.github.javaparser.resolution.declarations.ResolvedMethodDeclaration;
import com.github.javaparser.resolution.declarations.ResolvedValueDeclaration;
import com.github.javaparser.resolution.types.ResolvedType;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Rewrites Java source code to add or replace constructor parameters, method
 * invocations and more.
 */
public class JavaRewriter {

    private static final String UNKNOWN_WHERE = "Unknown where: ";
    private static final String CHART_SERIES_CLASS = "ChartSeries";
    private static final String ADD_CLASS_NAMES_METHOD = "addClassNames";
    private static final String SET_SPACING_METHOD = "setSpacing";
    private static final String ADD_ITEM_METHOD = "addItem";

    /**
     * A code snippet to be inserted into the source code.
     *
     * @param code
     *            the code snippet
     */
    public record Code(String code) {
    }

    /**
     * Holder for a setter name and associated value
     */
    public record SetterAndValue(String setter, Object value) {
    }

    /**
     * Information about extracting an inline variable to local variable
     */
    public record ExtractInlineVariableResult(BlockStmt blockStmt, String newVariableName, int index) {
    }

    /**
     * Information about a renamed variable.
     *
     * @param variableRenamedTo
     *            the new name for the variable
     */
    public record ReplaceResult(String variableRenamedTo) {

    }

    /**
     * The result of a duplicate operation
     *
     * @param nameMapping
     *            a map from old component name to new component name
     * @param childAddCalls
     *            a list of add calls from the parent to ddd children
     * @param variableDeclaration
     *            the variable declaration of the new component when it is declared
     *            as local variable to manage children attachment in constructor
     * @param assignExpr
     *            the assign expression of the new component when it is declared as
     *            field to manage children attachment in constructor
     */
    public record DuplicateInfo(Map<String, String> nameMapping, List<MethodCallExpr> childAddCalls,
            VariableDeclarationExpr variableDeclaration, AssignExpr assignExpr) {
    }

    public record AddTemplateOptions(boolean javaFieldsForLeafComponents, boolean useIdMappedFields, String methodName,
            AddReplace addReplace) {
        /**
         * Constructs AddTemplateOptions with <code>null</code> method name and
         * <code>null</code> addReplace
         *
         * @param javaFieldsForLeafComponents
         */
        public AddTemplateOptions(boolean javaFieldsForLeafComponents) {
            this(javaFieldsForLeafComponents, false, null, null);
        }

    }

    public record MoveTemplateOptions(String methodName, AddReplace addReplace) {
    }

    public enum AddReplace {
        ADD, REPLACE;

        public static AddReplace find(String str) {
            if (str == null) {
                return null;
            }
            return AddReplace.valueOf(str.toUpperCase(Locale.ENGLISH));
        }
    }

    /**
     * Where to add a component
     */
    public enum Where {
        BEFORE, APPEND
    }

    public enum AlignmentMode {
        ALIGN_ITEMS, SELF_HORIZONTALLY, SELF_VERTICALLY
    }

    /**
     * Replaces a constructor parameter (if it is mapped to the given setter
     * function) or a function call in the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function to replace or add, if the constructor
     *            parameter is not found
     * @param value
     *            the new value for the constructor parameter or function call, or
     *            {@code null} to remove the function call
     * @return the new variable name if the variable name was auto-generated and
     *         changed, otherwise null
     */
    public ReplaceResult replaceFunctionCall(ComponentInfo componentInfo, String function, Object value) {
        String currentText = null;
        String currentLabel = null;

        boolean affectsVariableName = JavaRewriterUtil.functionAffectsVariableName(function);
        boolean variableNameAutoGenerated = false;
        if (affectsVariableName) {
            Object currentTextProperty = getPropertyValue(componentInfo, "text");
            Object currentLabelProperty = getPropertyValue(componentInfo, "label");
            currentText = currentTextProperty instanceof String text ? text : null;
            currentLabel = currentLabelProperty instanceof String label ? label : null;
            variableNameAutoGenerated = JavaRewriterUtil.isVariableNameAutoGenerated(componentInfo, currentText,
                    currentLabel);
        }

        boolean replaced = replaceConstructorParam(componentInfo, function, value);
        if (!replaced) {
            if (value != null && value.getClass().isArray()) {
                replaceOrAddCall(componentInfo, function, (Object[]) value);
            } else {
                replaceOrAddCall(componentInfo, function, value);
            }

        }

        if (variableNameAutoGenerated) {
            String newText = function.equals("setText") && value instanceof String string ? string : null;
            String newLabel = function.equals("setLabel") && value instanceof String string ? string : null;
            // If we replace a static string with a translations call, we don't want to
            // rename the variable
            if (newText != null || newLabel != null) {
                return new ReplaceResult(JavaRewriterUtil.regenerateVariableName(componentInfo, newText, newLabel));
            }
        }
        return new ReplaceResult(null);
    }

    /**
     * Replaces a constructor parameter (if it is mapped to the given setter
     * function) in the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the setter function
     * @param value
     *            the new value for the constructor parameter
     * @return {@code true} if the replacement was successful, {@code false}
     *         otherwise
     */
    private boolean replaceConstructorParam(ComponentInfo componentInfo, String function, Object value) {
        ObjectCreationExpr objectCreationExpr = componentInfo.getCreateInfoOrThrow().getObjectCreationExpr();
        Optional<Constructor<?>> constructor = Optional.empty();
        if (objectCreationExpr == null && componentInfo.routeConstructor() != null) {
            constructor = JavaRewriterUtil.findConstructor(componentInfo.type(), componentInfo.routeConstructor());
        } else if (objectCreationExpr != null) {
            constructor = JavaRewriterUtil.findConstructor(componentInfo.type(), objectCreationExpr);
        }
        if (constructor.isEmpty()) {
            return false;
        }

        // Find constructor based on number of arguments
        Optional<Map<Integer, String>> mappings = constructor.map(c -> ConstructorAnalyzer.get().getMappings(c));
        if (mappings.isPresent()) {
            Optional<Map.Entry<Integer, String>> mapping = mappings.get().entrySet().stream()
                    .filter(entry -> entry.getValue().equals(function)).findFirst();
            if (mapping.isPresent() && objectCreationExpr != null) {
                if (value != null || objectCreationExpr.getArguments().size() > 1) {
                    objectCreationExpr.setArgument(mapping.get().getKey(), JavaRewriterUtil.toExpression(value));
                } else {
                    objectCreationExpr.getArguments().remove(mapping.get().getKey().intValue());
                }
                return true;
            }
        }
        return false;
    }

    /**
     * Adds the given service to the constructor of the class where the given
     * component is configured.
     * <p>
     * Optionally adds a field for the service, if the component is not configured
     * in the constructor
     *
     * @param component
     *            the reference component
     * @param serviceClassName
     *            the class name of the service class to add
     * @return the name of the added field, if a field was added, otherwise the name
     *         of the added parameter
     */
    private String addServiceToConstructor(ComponentInfo component, String serviceClassName) {
        boolean addServiceField = !JavaRewriterUtil.isObjectCreationExprInConstructor(component);
        String suggestedParameterName = SharedUtil.firstToLower(Util.getSimpleName(serviceClassName));
        return addConstructorParameterToClass(component, serviceClassName, suggestedParameterName, addServiceField);
    }

    /**
     * Adds a parameter of the given type to the constructor for the class
     * containing the given component.
     * <p>
     * Does not add another parameter, if a parameter of the given type is already
     * present.
     * <p>
     * Optionally creates a new field for the parameter and copy the value to the
     * field.
     *
     * @param referenceComponent
     *            a component instance created inside the class - used to find the
     *            class and its constructor
     * @param parameterTypeName
     *            the type of parameter
     * @param suggestedParameterName
     *            a suggested name for the parameter
     * @param storeAsField
     *            true to create a field and copy the parameter value there, false
     *            to add the parameter to the constructor only
     * @return the name of the added field, if storeAsField is true, otherwise the
     *         name of the added parameter
     */
    public String addConstructorParameterToClass(ComponentInfo referenceComponent, String parameterTypeName,
            String suggestedParameterName, boolean storeAsField) {
        CompilationUnit compilationUnit = referenceComponent.getCreateLocationCompilationUnitOrThrowIfNull();

        String parameterTypeSimpleName = Util.getSimpleName(parameterTypeName);

        Node ref;
        if (referenceComponent.routeConstructor() != null) {
            ref = referenceComponent.routeConstructor();
        } else {
            ref = referenceComponent.getCreateInfoOrThrow().getObjectCreationExpr();
        }
        TypeDeclaration<?> typeDeclaration = JavaRewriterUtil.findAncestorOrThrow(ref, TypeDeclaration.class);
        if (typeDeclaration.getConstructors().size() > 1) {
            throw new IllegalArgumentException("Cannot add constructor parameter to class with multiple constructors");
        }

        ConstructorDeclaration constructor;
        if (typeDeclaration.getConstructors().isEmpty()) {
            constructor = typeDeclaration.addConstructor();
        } else {
            constructor = typeDeclaration.getConstructors().get(0);
        }

        String parameterName = constructor.getParameterByType(parameterTypeSimpleName).map(Parameter::getNameAsString)
                .orElse(null);
        if (parameterName == null) {
            parameterName = JavaRewriterUtil.findFreeVariableName(suggestedParameterName, constructor.getBody());
            constructor.addParameter(parameterTypeSimpleName, parameterName);
            compilationUnit.addImport(parameterTypeName.replace("$", "."));
        }

        if (storeAsField) {
            List<FieldDeclaration> existingFields = typeDeclaration.getFields();
            ConstructorDeclaration firstConstructor = typeDeclaration.getConstructors().get(0);
            // Try to keep order so that the new field is added after existing fields, or
            // before the first constructor
            int fieldPos;
            if (!existingFields.isEmpty()) {
                fieldPos = typeDeclaration.getMembers().indexOf(existingFields.get(existingFields.size() - 1)) + 1;
            } else {
                fieldPos = typeDeclaration.getMembers().indexOf(firstConstructor);
            }
            String fieldName = JavaRewriterUtil.findFreeVariableName(parameterName,
                    JavaRewriterUtil.findAncestorOrThrow(typeDeclaration, ClassOrInterfaceDeclaration.class));
            FieldDeclaration newField = typeDeclaration.addField(parameterTypeSimpleName, fieldName,
                    Modifier.Keyword.PRIVATE, Modifier.Keyword.FINAL);
            typeDeclaration.getMembers().remove(newField);
            typeDeclaration.getMembers().add(fieldPos, newField);

            // Assigning to the field must happen after any super() call (cannot add any
            // code before that)
            // but before calling any other methods, as those might use the value
            BlockStmt body = constructor.getBody();
            int insertIndex = 0;
            if (body.getStatements().isNonEmpty()
                    && body.getStatement(0) instanceof ExplicitConstructorInvocationStmt) {
                insertIndex++;
            }

            body.addStatement(insertIndex, new AssignExpr(new FieldAccessExpr(new ThisExpr(), fieldName),
                    new NameExpr(parameterName), AssignExpr.Operator.ASSIGN));
            return fieldName;
        }

        return parameterName;

    }

    /**
     * Adds a function call to the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function to add
     * @param parameters
     *            parameters for the function
     */
    public void addCall(ComponentInfo componentInfo, String function, Object... parameters) {
        doReplaceOrAddCall(componentInfo, function, false, parameters);
    }

    /**
     * Replaces a function call in the source code, if found, otherwise adds the
     * function call.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function call to add or replace
     * @param parameters
     *            new parameters for the function
     */
    public void replaceOrAddCall(ComponentInfo componentInfo, String function, Object... parameters) {
        doReplaceOrAddCall(componentInfo, function, true, parameters);
    }

    /**
     * Removes all property set calls or references in constructors initializations
     *
     * @param componentInfo
     *            the component to modify
     * @param setter
     *            the name of the property setter.
     */
    public void removePropertySetter(ComponentInfo componentInfo, String setter) {
        boolean replaced = replaceConstructorParam(componentInfo, setter, null);
        if (!replaced) {
            removeCalls(componentInfo, setter);
        }
    }

    /**
     * Removes all calls to the given function from the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function to remove parameter is not found
     */
    public void removeCalls(ComponentInfo componentInfo, String function) {
        JavaRewriterUtil.findMethodCallStatements(componentInfo, function).forEach(JavaRewriterUtil::removeStatement);
        JavaRewriterUtil.findMethodCallNonStatements(componentInfo, function)
                .forEach(JavaRewriterUtil::removeStatement);
    }

    private void doReplaceOrAddCall(ComponentInfo componentInfo, String function, boolean replace,
            Object... parameters) {

        List<String> addImportClassNames = new ArrayList<>();
        List<Expression> parameterExpressions = new ArrayList<>();
        for (Object parameter : parameters) {
            if (parameter instanceof Enum<?>) {
                addImportClassNames.add(parameter.getClass().getName());
            } else if (parameter instanceof BigDecimal) {
                addImportClassNames.add(BigDecimal.class.getName());
            } else if (parameter instanceof BigInteger) {
                addImportClassNames.add(BigInteger.class.getName());
            }
            parameterExpressions.add(JavaRewriterUtil.toExpression(parameter));
        }
        MethodCallExpr functionCall = JavaRewriterUtil.findMethodCallStatements(componentInfo, function).findFirst()
                .orElse(null);
        List<Expression> newParameterExpressions = new ArrayList<>();

        List<List<JavaReflectionUtil.ParameterTypeInfo>> allMethodParameterTypes = JavaReflectionUtil
                .getAllMethodParameterTypes(componentInfo.type(), function);
        Optional<List<JavaReflectionUtil.ParameterTypeInfo>> optionalParameterTypeInfos = allMethodParameterTypes
                .stream()
                .filter(parameterTypes -> parameterTypes.size() == 1 && parameterTypes.get(0).type() != null
                        && parameterTypes.get(0).type().isArray() && !parameterTypes.get(0).type().isVarArgs())
                .findFirst();

        if (optionalParameterTypeInfos.isPresent()) {
            JavaReflectionUtil.ParameterTypeInfo parameterTypeInfo = optionalParameterTypeInfos.get().get(0);
            String parameterClassName = parameterTypeInfo.type().typeName().replace("[]", "");
            Class<?> aClass = JavaReflectionUtil.getClass(parameterClassName);
            addImportClassNames.add(aClass.getName());
            ArrayCreationExpr arrayCreationExpr = new ArrayCreationExpr();
            arrayCreationExpr.setElementType(aClass.getSimpleName());
            arrayCreationExpr.setInitializer(new ArrayInitializerExpr(NodeList.nodeList(parameterExpressions)));
            newParameterExpressions.add(arrayCreationExpr);
        } else {
            newParameterExpressions.addAll(parameterExpressions);
        }
        if (replace && functionCall != null) {
            functionCall.setArguments(NodeList.nodeList(newParameterExpressions));
            return;
        }

        MethodCallExpr methodCallExpr = JavaRewriterUtil.addFunctionCall(componentInfo, function,
                newParameterExpressions);
        CompilationUnit compilationUnit = JavaRewriterUtil.findAncestor(methodCallExpr, CompilationUnit.class);
        if (compilationUnit != null) {
            addImportClassNames.forEach(className -> JavaRewriterUtil.addImport(compilationUnit, className));
        }
    }

    /**
     * Gets the (active) value of a property of a component.
     *
     * <p>
     * The property value is determined by looking for a setter method call in the
     * source code. If the property is not set using a setter, the constructor is
     * checked.
     *
     * <p>
     * If the property is not set using a setter or in the constructor, {@code null}
     * is returned.
     *
     * <p>
     * If the property is set using a method call, the method call expression is
     * returned.
     *
     * @param componentInfo
     *            the component to get the property value from
     * @param property
     *            the property name
     * @return the property value, or null if the property is not set
     */
    public Object getPropertyValue(ComponentInfo componentInfo, String property) {
        String setterName = JavaRewriterUtil.getSetterName(property, componentInfo.type(), false);
        List<MethodCallExpr> functionCalls = JavaRewriterUtil.findMethodCallStatements(componentInfo);
        List<MethodCallExpr> candidates = functionCalls.stream().filter(m -> m.getNameAsString().equals(setterName))
                .toList();
        if (!candidates.isEmpty()) {
            // If there would be multiple calls, the last one is the effective
            // one
            MethodCallExpr setterCall = candidates.get(candidates.size() - 1);
            Expression arg = setterCall.getArguments().get(0);

            ResolvedType expectedType = null;
            try {
                expectedType = setterCall.resolve().getParam(0).getType();
            } catch (Exception e) {
                getLogger().debug("Unable to resolve setter parameter type for {}.{}", componentInfo.type().getName(),
                        setterName, e);
            }
            return JavaRewriterUtil.fromExpression(arg, expectedType);
        }

        // Try constructor
        ObjectCreationExpr createExpression = componentInfo.getCreateInfoOrThrow().getObjectCreationExpr();
        if (createExpression != null) {
            Optional<Constructor<?>> maybeConstructor = JavaRewriterUtil.findConstructor(componentInfo.type(),
                    createExpression);
            if (maybeConstructor.isPresent()) {
                Constructor<?> constructor = maybeConstructor.get();
                for (int i = 0; i < constructor.getParameterCount(); i++) {
                    String mappedProperty = JavaRewriterUtil.getMappedProperty(constructor, i);
                    if (setterName.equals(mappedProperty)) {
                        return JavaRewriterUtil.fromExpression(createExpression.getArgument(i),
                                createExpression.resolve().getParam(i).getType());
                    }
                }
            }
        }
        return null;
    }

    /**
     * Gets the (active) styles of a component.
     *
     * @param componentInfo
     *            the component to get the styles of
     * @return the styles, as a list of style names and values
     */
    public List<JavaStyleRewriter.StyleInfo> getStyles(ComponentInfo componentInfo) {
        return JavaStyleRewriter.getStyles(componentInfo);
    }

    /**
     * Sets the given inline style on the given component, replacing an existing
     * style property if present.
     *
     * @param component
     *            the component to set the style on
     * @param property
     *            the style property to set
     * @param value
     *            the style value to set or null to remove the style
     */
    public void setStyle(ComponentInfo component, String property, String value) {
        JavaStyleRewriter.setStyle(component, property, value);
    }

    /**
     * Sets sizing properties of the given component using the Style API. Replaces
     * existing ones and removes the keys which have null values. <br/>
     * For removal, it also looks for Component API to remove such
     * <code>button.setWidth("")</code> calls.
     *
     * @param componentInfo
     *            the component to set style
     * @param changes
     *            changes applied. Having null value for a key means removal,
     *            otherwise update/add applies.
     */
    public void setSizing(ComponentInfo componentInfo, Map<String, String> changes) {
        JavaStyleRewriter.setSizing(componentInfo, changes);
    }

    /**
     * Deletes a component from the source code.
     *
     * @param componentInfo
     *            the component to delete
     * @return {@code true} if the deletion was successful, {@code false} otherwise
     */
    public boolean delete(ComponentInfo componentInfo) {
        if (deleteMethodCallWhenRemovingTheComponent(componentInfo)) {
            return true;
        }
        List<MethodCallExpr> functionCalls = JavaRewriterUtil.findMethodCallStatements(componentInfo);
        List<MethodCallExpr> otherMethodCalls = JavaRewriterUtil.findMethodCallNonStatements(componentInfo);

        List<Expression> parameterUsages = JavaRewriterUtil.findParameterUsage(componentInfo);
        List<ReturnStmt> returnStmts = JavaRewriterUtil.findReturnStatements(componentInfo);

        Optional<Range> attachRange = componentInfo.componentAttachInfoOptional()
                .filter(attach -> componentInfo.createAndAttachLocationsAreInSameFile())
                .map(ComponentAttachInfo::getAttachCall).map(AttachExpression::getNode).flatMap(Node::getRange);
        if (attachRange.isPresent()) {
            parameterUsages = parameterUsages.stream().filter(parameter -> parameter.getRange().isPresent()
                    && !parameter.getRange().get().overlapsWith(attachRange.get())).toList();
        }

        List<MethodCallExpr> removeMethodCalls = new ArrayList<>();
        parameterUsages.forEach(paramExpression -> this.deleteParameterUsage(paramExpression, removeMethodCalls));

        functionCalls.forEach(expr -> JavaRewriterUtil.findAncestorOrThrow(expr, Statement.class).remove());
        if (componentInfo.getCreateInfoOrThrow().getFieldDeclaration() != null) {
            componentInfo.getCreateInfoOrThrow().getFieldDeclaration().remove();
            if (componentInfo.getCreateInfoOrThrow().getFieldDeclarationAndAssignment() == null) {
                // Instance created elsewhere, i.e.
                // TextField foo;
                // ...
                // foo = new TextField();
                JavaRewriterUtil.removeStatement(componentInfo.getCreateInfoOrThrow().getObjectCreationExpr());
            }
        } else if (componentInfo.getCreateInfoOrThrow().getLocalVariableDeclarator() != null) {
            JavaRewriterUtil.removeStatement(componentInfo.getCreateInfoOrThrow().getLocalVariableDeclarator());
        }

        otherMethodCalls.forEach(methodCall -> {
            if (!JavaRewriterUtil.removeFromStringConcatenation(methodCall)
                    && !JavaRewriterUtil.removeFromCondition(methodCall)) {
                JavaRewriterUtil.removeStatement(methodCall);
            }
        });

        returnStmts.forEach(Node::remove);

        removeAttachCall(componentInfo);
        BlockStmt createScope = componentInfo.getCreateInfoOrThrow().getComponentCreateScope();
        Optional<MethodDeclaration> createMethod = JavaRewriterUtil.getMethodIfEmpty(createScope);
        createMethod.ifPresent(Node::remove);

        return true;
    }

    /***
     * Deletes the method call rather deleting statements inside the method call
     * when criteria are matched. This method handles the cases where a component is
     * created in a method and returned, and the method is used somewhere else to
     * create the component. Factory pattern is one of the simple use-cases
     * <hr />
     * Also, removes the method declaration if there is no usage in the class.
     *
     * @param componentInfo
     *            Component that will be deleted.
     * @return True if criteria are matched and method call is removed, false
     *         otherwise.
     */
    private boolean deleteMethodCallWhenRemovingTheComponent(ComponentInfo componentInfo) {
        if (componentInfo.componentAttachInfoOptional().isEmpty()) {
            return false;
        }
        if (componentInfo.componentCreateInfoOptional().isEmpty()) {
            return false;
        }
        ComponentCreateInfo componentCreateInfo = componentInfo.componentCreateInfoOptional().get();
        if (componentCreateInfo.getFieldDeclaration() != null) {
            return false;
        }
        if (componentCreateInfo.getLocalVariableDeclarator() == null) {
            // should work only for local variables created in a method that returns the
            // component
            return false;
        }
        ComponentAttachInfo componentAttachInfo = componentInfo.componentAttachInfoOptional().get();
        if (componentAttachInfo.getAttachCall().getMethodCallExpression() == null) {
            return false;
        }
        MethodDeclaration methodDeclarationContainingComponentCreateExpr = JavaRewriterUtil
                .findAncestor(componentCreateInfo.getLocalVariableDeclarator(), MethodDeclaration.class);
        if (methodDeclarationContainingComponentCreateExpr == null) {
            return false;
        }
        if (!methodDeclarationContainingComponentCreateExpr.getType().isClassOrInterfaceType()) {
            return false;
        }
        if (methodDeclarationContainingComponentCreateExpr.getBody().isEmpty()) {
            return false;
        }
        String methodReturnTypeQualifiedClassName = methodDeclarationContainingComponentCreateExpr.getType().resolve()
                .describe();
        if (!methodReturnTypeQualifiedClassName.equals(componentInfo.type().getName())) {
            return false;
        }
        BlockStmt blockStmt = methodDeclarationContainingComponentCreateExpr.getBody().get();
        boolean returnStmtsAreMatchedWithLocalVarName = blockStmt.getStatements().stream()
                .filter(Statement::isReturnStmt).map(ReturnStmt.class::cast).allMatch(returnStmt -> {
                    if (returnStmt.getExpression().isEmpty()) {
                        return false;
                    }
                    Expression expression = returnStmt.getExpression().get();
                    return expression.isNameExpr() && expression.asNameExpr().getName().toString()
                            .equals(componentCreateInfo.getLocalVariableName());
                });
        if (!returnStmtsAreMatchedWithLocalVarName) {
            return false;
        }
        // all statements are matched. Removing the attach call
        removeAttachCall(componentInfo);
        deleteMethodDeclarationIfNoLongerUsedInSameClass(methodDeclarationContainingComponentCreateExpr);
        return true;
    }

    /**
     * Deletes the method declaration if is not Protected or Public and also checks
     * the usage in the same class. If there is a usage in the same class or
     * modifiers are whether protected or public, it does not do anything.
     *
     * @param methodDeclaration
     *            declaration to check its usages and modifiers
     */
    private void deleteMethodDeclarationIfNoLongerUsedInSameClass(MethodDeclaration methodDeclaration) {
        AccessSpecifier accessSpecifier = methodDeclaration.getAccessSpecifier();

        if (AccessSpecifier.PROTECTED == accessSpecifier || AccessSpecifier.PUBLIC == accessSpecifier) {
            return;
        }
        ClassOrInterfaceDeclaration clazz = JavaRewriterUtil.findAncestorOrThrow(methodDeclaration,
                ClassOrInterfaceDeclaration.class);
        List<MethodCallExpr> list = clazz.findAll(MethodCallExpr.class).stream().filter(call -> {
            try {
                ResolvedMethodDeclaration resolved = call.resolve();
                return resolved.getName().equals(methodDeclaration.getNameAsString())
                        && methodDeclaration.getParentNode().isPresent()
                        && resolved.declaringType().getQualifiedName()
                                .equals(((ClassOrInterfaceDeclaration) methodDeclaration.getParentNode().get())
                                        .resolve().getQualifiedName());
            } catch (Exception e) {
                getLogger().debug("Could not resolve method call", e);
                return false; // unresolved symbol
            }
        }).toList();
        if (list.isEmpty()) {
            methodDeclaration.remove();
        }

    }

    private void deleteParameterUsage(Expression parameterUsageExpr, List<MethodCallExpr> removeMethodCalls) {
        if (parameterUsageExpr.findAncestor(n -> {
            if (n.getScope().isPresent()) {
                return removeMethodCalls.contains(n.getScope().get());
            } else {
                return false;
            }
        }, MethodCallExpr.class).isPresent()) {
            // If the method scope is already removed, we skip this expression
            return;
        }
        if (parameterUsageExpr.getParentNode().isEmpty()) {
            return;
        }
        Node parentNode = parameterUsageExpr.getParentNode().get();
        if (parentNode instanceof MethodCallExpr) {
            MethodCallExpr methodCallExpr = (MethodCallExpr) parameterUsageExpr.getParentNode().get();
            Optional<Expression> scopeOptional = methodCallExpr.getScope();
            Optional<String> classNameOpt;
            if (scopeOptional.isPresent() && scopeOptional.get() instanceof NameExpr nameExpr) {
                CompilationUnit compilationUnit = JavaRewriterUtil.findAncestor(nameExpr, CompilationUnit.class);
                if (compilationUnit == null) {
                    // meaning that ancestor statement of this node has been deleted.
                    return;
                }
                classNameOpt = JavaRewriterUtil.getQualifiedClassName(nameExpr.resolve());
            } else {
                classNameOpt = JavaRewriterUtil.findAncestor(methodCallExpr, ClassOrInterfaceDeclaration.class)
                        .getFullyQualifiedName();
            }
            if (classNameOpt.isEmpty()) {
                throw new IllegalArgumentException("Could not find source class");
            }
            int argumentPosition = methodCallExpr.getArgumentPosition(parameterUsageExpr);
            boolean arrayArgument = JavaReflectionUtil.isArrayArgument(classNameOpt.get(),
                    methodCallExpr.getNameAsString(), argumentPosition);
            if (arrayArgument) {
                JavaRewriterUtil.removeArgumentCalls(methodCallExpr, List.of(parameterUsageExpr), true);
            } else if (methodCallExpr.getArguments().size() == 1) {
                JavaRewriterUtil.removeStatement(methodCallExpr);
            } else {
                throw new IllegalArgumentException("Cannot handle " + methodCallExpr.getNameAsString() + " method");
            }
        } else if (parentNode instanceof ExpressionStmt) {
            throw new IllegalArgumentException("Unable to delete " + parentNode.toString());
        }
    }

    private Optional<Expression> removeAttachCall(ComponentInfo componentInfo) {
        Optional<Expression> addArgument = JavaRewriterUtil.getAttachArgument(componentInfo);
        addArgument.ifPresent(Expression::remove);
        Optional<ComponentAttachInfo> componentAttachInfoOptional = componentInfo.componentAttachInfoOptional();
        if (componentAttachInfoOptional.isPresent()) {
            ComponentAttachInfo componentAttachInfo = componentAttachInfoOptional.get();
            if (componentAttachInfo.getAttachCall() != null
                    && componentAttachInfo.getAttachCall().getNodeWithArguments().getArguments().isEmpty()) {
                JavaRewriterUtil.removeStatement(componentAttachInfo.getAttachCall().getNode());
            }
        }
        return addArgument;
    }

    /**
     * Moves a component in the source code.
     *
     * @param component
     *            the component to move
     * @param container
     *            the new container for the component, if where is Where.APPEND.
     * @param reference
     *            the reference component to move the component before, if where is
     *            Where.BEFORE.
     * @param where
     *            where to move the component
     */
    public void moveComponent(ComponentInfo component, ComponentInfo container, ComponentInfo reference, Where where,
            MoveTemplateOptions options) {
        if (container == null) {
            throw new IllegalArgumentException("Container component must be non-null");
        }
        if (component.equals(container) || component.equals(reference) || container.equals(reference)) {
            throw new IllegalArgumentException("Component, container and reference must be different");
        }
        // Remember previous location to use as fallback
        InsertionPoint previousLocation;
        if (component.componentAttachInfoOptional().isPresent()) {
            previousLocation = JavaRewriterUtil
                    .findLocationBefore(component.componentAttachInfoOptional().get().getAttachCall().getNode());
        } else {
            previousLocation = null;
        }
        removeAttachCall(component);

        List<MethodCallExpr> containerFunctionCalls = JavaRewriterUtil.findMethodCallStatements(container);
        String componentFieldOrVariableName = JavaRewriterUtil.getFieldOrVariableName(component);
        // For inline we add the full expression, otherwise the name reference
        var toAdd = componentFieldOrVariableName == null ? component.getCreateInfoOrThrow().getObjectCreationExpr()
                : new NameExpr(componentFieldOrVariableName);
        if (where == Where.APPEND) {
            if (reference != null) {
                throw new IllegalArgumentException("Reference component must be null when appending");
            }

            // Find the last add statement for the container and then add the
            // component after that, or then use the container create expression
            // if none of this options are available insert in the previous location
            Optional<MethodCallExpr> lastFunctionCall = JavaRewriterUtil.findLastFunctionCall(containerFunctionCalls,
                    "add", toAdd);
            InsertionPoint insertLocation = lastFunctionCall.map(JavaRewriterUtil::findLocationAfter).orElseGet(() -> {
                if (container.getCreateInfoOrThrow().getObjectCreationExpr() == null) {
                    if (previousLocation != null) {
                        return previousLocation;
                    } else {
                        throw new IllegalArgumentException("Could not find insert location for component in container");
                    }
                } else {
                    return JavaRewriterUtil.findLocationAfter(container.getCreateInfoOrThrow().getObjectCreationExpr());
                }
            });

            if (JavaRewriterUtil.findAncestor(component.getCreateInfoOrThrow().getObjectCreationExpr(),
                    BlockStmt.class) != null) {
                // We must ensure that the component is created before it is added
                InsertionPoint componentCreateLocation = JavaRewriterUtil
                        .findLocationAfter(component.getCreateInfoOrThrow().getObjectCreationExpr());
                if (componentCreateLocation.isAfter(insertLocation)) {
                    insertLocation = componentCreateLocation;
                }

            }
            var createInfo = container.getCreateInfoOrThrow();
            var containerVariableName = createInfo.getLocalVariableName() == null ? createInfo.getFieldName()
                    : createInfo.getLocalVariableName();
            if (containerVariableName == null && container.getCreateInfoOrThrow().getObjectCreationExpr() != null) {
                // Inline defined container
                ExtractInlineVariableResult extracted = JavaRewriterUtil
                        .extractInlineVariableToLocalVariable(container);
                if (extracted == null) {
                    throw new IllegalArgumentException("Could not extract inline variable to local variable");
                }
                containerVariableName = extracted.newVariableName();
            }
            Expression scope = null;
            if (containerVariableName != null) {
                // Everything else but a class that extends a layout
                scope = new NameExpr(containerVariableName);
            }

            if (scope == null && JavaRewriterUtil.isNodeInCompositeClass(
                    insertLocation.getBlock() != null ? insertLocation.getBlock() : insertLocation.getDeclaration())) {
                scope = new NameExpr("getContent()");
            }

            if (options != null && options.methodName != null) {
                Optional<MethodCallExpr> methodCallExprOptional = JavaRewriterUtil
                        .findMethodCallStatements(container, options.methodName()).findFirst();
                MethodCallExpr methodCallExpr;
                if (methodCallExprOptional.isEmpty()) {
                    methodCallExpr = new MethodCallExpr(
                            containerVariableName != null ? new NameExpr(containerVariableName) : scope,
                            options.methodName);
                    insertLocation.add(new ExpressionStmt(methodCallExpr));
                } else {
                    methodCallExpr = methodCallExprOptional.get();
                }
                if (options.addReplace == AddReplace.ADD) {
                    boolean arrayArgument = JavaReflectionUtil.isArrayArgument(container.type().getName(),
                            methodCallExpr.getNameAsString(), 0);
                    if (arrayArgument) {
                        methodCallExpr.getArguments().add(toAdd);
                    } else {
                        // if method argument is not an array, then wrap them with Div
                        List<Expression> currentArgs = new ArrayList<>(methodCallExpr.getArguments().stream().toList());
                        JavaRewriterUtil.removeArgumentCalls(methodCallExpr, currentArgs, false);
                        Expression expressionToAddAsArgs;
                        if (!currentArgs.isEmpty()) {
                            CompilationUnit compilationUnit = container.getAttachLocationCompilationUnitOrThrowIfNull();
                            JavaRewriterUtil.addImport(compilationUnit, "com.vaadin.flow.component.html.Div");
                            ObjectCreationExpr wrapperDivCreationExpr = new ObjectCreationExpr();
                            wrapperDivCreationExpr.setType("Div");
                            currentArgs.forEach(wrapperDivCreationExpr::addArgument);
                            wrapperDivCreationExpr.addArgument(toAdd);
                            expressionToAddAsArgs = wrapperDivCreationExpr;
                        } else {
                            expressionToAddAsArgs = toAdd;
                        }
                        methodCallExpr.addArgument(expressionToAddAsArgs);
                    }
                } else if (options.addReplace == AddReplace.REPLACE) {
                    // removes all method args and appends the new one to given attach call method
                    List<Expression> currentArgs = new ArrayList<>(methodCallExpr.getArguments().stream().toList());
                    JavaRewriterUtil.removeArgumentCalls(methodCallExpr, currentArgs, false);
                    methodCallExpr.getArguments().add(toAdd);
                }
            } else {
                insertLocation.add(new ExpressionStmt(JavaRewriterUtil.createMethodCall(scope, "add", toAdd)));
            }

        } else if (where == Where.BEFORE) {
            if (reference == null) {
                throw new IllegalArgumentException("Reference component must be non-null when moving before");
            }

            Optional<Expression> referenceAddArgument = JavaRewriterUtil.findReference(
                    reference.getAttachCallInSameFileOrThrow().getNodeWithArguments().getArguments(), reference);

            if (referenceAddArgument.isEmpty()) {
                throw new IllegalArgumentException("Reference component not found in the add call");
            }
            int refAddIndex = reference.getAttachCallInSameFileOrThrow().getNodeWithArguments().getArguments()
                    .indexOf(referenceAddArgument.get());
            reference.getAttachCallInSameFileOrThrow().getNodeWithArguments().getArguments().add(refAddIndex, toAdd);

            // We can now end up with having "add" before the component
            // constructor and setters, e.g.

            // Input input1 = new Input();
            // add(input3, input1);
            // Input input2 = new Input();
            // add(input2);
            // Input input3 = new Input();

            // We move the add call down after all functions
            // BUT there can also be add calls between the current
            // add location and where it should end up.
            // If so, they need to be moved also
            // We then get

            // Input input1 = new Input();
            // Input input2 = new Input();
            // Input input3 = new Input();
            // add(input3, input1);
            // add(input2);

            List<MethodCallExpr> componentFunctionCalls = JavaRewriterUtil.findMethodCallStatements(component);
            ObjectCreationExpr objectCreationExpr = component.getCreateInfoOrThrow().getObjectCreationExpr();
            MethodCallExpr lastFunctionCall = componentFunctionCalls.isEmpty() ? null
                    : componentFunctionCalls.get(componentFunctionCalls.size() - 1);

            BlockStmt objectCreationBlock = JavaRewriterUtil.findAncestor(objectCreationExpr, BlockStmt.class);

            BlockStmt insertBlock = null;
            boolean inDifferentBlocks = JavaRewriterUtil.expressionsInDifferentBlocks(referenceAddArgument.get(),
                    objectCreationExpr.asObjectCreationExpr());
            Integer finalAddLocation = null;
            if (inDifferentBlocks && component.getCreateInfoOrThrow().getFieldName() == null) {
                insertBlock = JavaRewriterUtil.findAncestor(referenceAddArgument.get(), BlockStmt.class);
                if (insertBlock == null) {
                    throw new IllegalArgumentException("Could not find insert block");
                }
                // component is moved to another method.
                // we should also move all statements related to component to new component
                finalAddLocation = JavaRewriterUtil.findBlockStatementIndex(referenceAddArgument.get());
                InsertionPoint insertionPoint = new InsertionPoint(insertBlock, finalAddLocation - 1);
                // now moving the object creation expression
                Expression expressionToMove;
                VariableDeclarationExpr variableDeclarationExpr = JavaRewriterUtil.findAncestor(objectCreationExpr,
                        VariableDeclarationExpr.class);
                // selecting initializer expression to move
                expressionToMove = Objects.requireNonNullElse(variableDeclarationExpr, objectCreationExpr);
                // removing the original
                JavaRewriterUtil.removeStatement(expressionToMove);
                insertionPoint.add(new ExpressionStmt(expressionToMove));
                // now moving the methods
                List<MethodCallExpr> methodCalls = JavaRewriterUtil.findMethodCalls(component);
                // TODO move parameters to the new method when it is only used for moved
                // component, or make them field variable or throw an exception

                // methods are moved
                methodCalls.forEach(methodCallExpr -> insertionPoint
                        .add(new ExpressionStmt(JavaRewriterUtil.clone(methodCallExpr))));
                methodCalls.forEach(JavaRewriterUtil::removeStatement);

            } else if (lastFunctionCall != null) {
                BlockStmt lastFunctionCallBlock = JavaRewriterUtil.findAncestorOrThrow(lastFunctionCall,
                        BlockStmt.class);
                insertBlock = lastFunctionCallBlock;
                int functionCallIndex = JavaRewriterUtil.findBlockStatementIndex(lastFunctionCall) + 1;
                if (lastFunctionCallBlock == objectCreationBlock) {
                    int objectCreationIndex = JavaRewriterUtil.findBlockStatementIndex(objectCreationExpr) + 1;
                    finalAddLocation = Math.max(functionCallIndex, objectCreationIndex);
                } else {
                    finalAddLocation = functionCallIndex;
                }
            } else {
                if (objectCreationBlock != null) {
                    insertBlock = objectCreationBlock;
                    finalAddLocation = JavaRewriterUtil.findBlockStatementIndex(objectCreationExpr) + 1;
                }
            }

            int attachCallIndex = JavaRewriterUtil
                    .findBlockStatementIndex(reference.getAttachCallInSameFileOrThrow().getNode());

            List<MethodCallExpr> allAddCalls = containerFunctionCalls.stream()
                    .filter(m -> m.getNameAsString().equals(
                            reference.getAttachCallInSameFileOrThrow().getNodeWithSimpleName().getName().asString()))
                    .toList();
            if (finalAddLocation != null) {
                for (MethodCallExpr addCall : allAddCalls) {
                    int addCallIndex = JavaRewriterUtil.findBlockStatementIndex(addCall);
                    if (addCallIndex >= attachCallIndex && addCallIndex < finalAddLocation) {
                        Statement statement = JavaRewriterUtil.findAncestorOrThrow(addCall, Statement.class);
                        statement.remove();
                        insertBlock.addStatement(finalAddLocation - 1, statement);
                    }
                }
            }

        } else {
            throw new IllegalArgumentException(UNKNOWN_WHERE + where);
        }
    }

    public void duplicate(ComponentInfo component) {
        duplicate(component, true);
    }

    /**
     * Duplicates a component in the source code.
     *
     * @param component
     *            the component to duplicate
     * @param handleAdd
     *            true to automatically add the new component next to the old one,
     *            false to handle add calls like any other method call
     * @return a map from old component name to new component name
     */
    public DuplicateInfo duplicate(ComponentInfo component, boolean handleAdd) {
        if (component.routeConstructor() != null) {
            throw new IllegalArgumentException("Cannot duplicate a route class");
        }

        // Insert new component after the original component
        InsertionPoint insertionPoint = findInsertionPointForAppend(component);
        String oldName;
        List<MethodCallExpr> childAddCalls = new ArrayList<>();
        VariableDeclarationExpr variableDeclaration = null;
        AssignExpr assignExpr = null;

        String duplicatedName;
        var createInfo = component.getCreateInfoOrThrow();
        if (createInfo.getLocalVariableName() != null) {
            oldName = createInfo.getLocalVariableName();
            duplicatedName = JavaRewriterUtil.findFreeVariableName(createInfo.getLocalVariableName(),
                    insertionPoint.getBlock());
            VariableDeclarator newLocalVariable = JavaRewriterUtil.clone(createInfo.getLocalVariableDeclarator());
            newLocalVariable.setName(duplicatedName);
            variableDeclaration = new VariableDeclarationExpr(newLocalVariable);
            JavaRewriterUtil.appendExpressionAsNextSiblingInBlockAncestor(createInfo.getLocalVariableDeclarator(),
                    new ExpressionStmt(variableDeclaration));
            insertionPoint.incrementIndexByOne();
        } else if (createInfo.getFieldName() != null) {
            oldName = createInfo.getFieldName();
            duplicatedName = insertionPoint.getFreeVariableName(createInfo.getFieldName());
            FieldDeclaration newField;
            if (createInfo.getFieldDeclarationAndAssignment() != null) {
                newField = JavaRewriterUtil.clone(createInfo.getFieldDeclarationAndAssignment());
                if (newField.getVariables().isNonEmpty()) {
                    variableDeclaration = new VariableDeclarationExpr(newField.getVariable(0));
                }
            } else {
                newField = JavaRewriterUtil.clone(createInfo.getFieldDeclaration());
                if (newField.getVariables().size() > 1) {
                    newField.getVariables()
                            .removeIf(variableDeclarator -> !variableDeclarator.getNameAsString().equals(oldName));
                }
                // Assignment, e.g. button = new Button();
                assignExpr = JavaRewriterUtil.clone(createInfo.getAssignmentExpression());
                assignExpr.setTarget(new NameExpr(duplicatedName));
                JavaRewriterUtil.appendExpressionAsNextSiblingInBlockAncestor(createInfo.getAssignmentExpression(),
                        new ExpressionStmt(assignExpr));
            }
            newField.getVariable(0).setName(duplicatedName);
            JavaRewriterUtil.addFieldAfter(newField, createInfo.getFieldDeclaration());
        } else {
            duplicatedName = null;
            // Inline
            oldName = null;
        }

        // Duplicate method calls
        if (duplicatedName != null) {
            List<MethodCallExpr> calls = JavaRewriterUtil.findMethodCallStatements(component);
            for (MethodCallExpr call : calls) {
                ExpressionStmt duplicatedStmt;

                VariableDeclarator ancestorVariableDeclarator = JavaRewriterUtil.findAncestor(call,
                        VariableDeclarator.class);
                // for method calls that is assigned to a variable e.g. MenuItem viewMenuItem =
                // menubar.addItem("View");
                if (ancestorVariableDeclarator != null) {
                    VariableDeclarator clone = JavaRewriterUtil.clone(ancestorVariableDeclarator);
                    clone.getInitializer().filter(Expression::isMethodCallExpr).map(MethodCallExpr.class::cast)
                            .ifPresent(methodCallExpr -> JavaRewriterUtil.setNameExprScope(methodCallExpr,
                                    new NameExpr(duplicatedName)));

                    clone.setName(
                            JavaRewriterUtil.findFreeVariableName(clone.getNameAsString(), insertionPoint.getBlock()));
                    duplicatedStmt = new ExpressionStmt(new VariableDeclarationExpr(clone));
                } else {
                    // for method calls without variable assignment e.g. button.setId();
                    MethodCallExpr newCall = JavaRewriterUtil.clone(call);
                    JavaRewriterUtil.setNameExprScope(newCall, new NameExpr(duplicatedName));
                    duplicatedStmt = new ExpressionStmt(newCall);
                    if (newCall.getNameAsString().equals("add")) {
                        childAddCalls.add(newCall);
                    }
                }

                insertionPoint.add(duplicatedStmt);

            }

            List<Expression> parameterUsages = JavaRewriterUtil.findParameterUsage(component);
            // excluding attach call
            Optional<Range> attachCallRangeOptional = component.getAttachCallInSameFileOrThrow().getNodeWithRange()
                    .getRange();
            if (attachCallRangeOptional.isPresent()) {
                Range attachRange = attachCallRangeOptional.get();
                parameterUsages = parameterUsages.stream().filter(parameter -> parameter.getRange().isPresent()
                        && !parameter.getRange().get().overlapsWith(attachRange)).toList();
            }
            for (Expression parameterUsage : parameterUsages) {
                MethodCallExpr methodCallExpr = JavaRewriterUtil.findAncestorOrThrow(parameterUsage,
                        MethodCallExpr.class);
                int argumentPosition = methodCallExpr.getArgumentPosition(parameterUsage);
                boolean addAsArg = false;
                Optional<Expression> scope = JavaRewriterUtil.getScopeIgnoreComposite(component, methodCallExpr);
                if (scope.isEmpty() || scope.get().isThisExpr()) {
                    Optional<String> classNameOpt = JavaRewriterUtil
                            .findAncestor(parameterUsage, ClassOrInterfaceDeclaration.class).getFullyQualifiedName();
                    if (classNameOpt.isEmpty()) {
                        throw new IllegalArgumentException("Could not find source class");
                    }
                    addAsArg = JavaReflectionUtil.isArrayArgument(classNameOpt.get(), methodCallExpr.getNameAsString(),
                            argumentPosition);
                }
                if (addAsArg) {
                    methodCallExpr.getArguments().add(argumentPosition + 1, new NameExpr(duplicatedName));
                } else {
                    MethodCallExpr clone = JavaRewriterUtil.clone(methodCallExpr);
                    clone.setArgument(argumentPosition, new NameExpr(duplicatedName));
                    insertionPoint.add(new ExpressionStmt(clone));
                }
            }
        }
        // Attach the new component
        if (handleAdd) {
            Expression componentAddArgument = JavaRewriterUtil.getAttachArgumentOrThrow(component);

            int addIndex;
            NodeWithArguments<?> expressionToAdd;
            Optional<Expression> parentNode = componentAddArgument.getParentNode().map(Expression.class::cast);
            if (parentNode.isPresent() && parentNode.get().isObjectCreationExpr()) {
                ObjectCreationExpr objectCreationExpr = parentNode.get().asObjectCreationExpr();
                addIndex = objectCreationExpr.getArguments().indexOf(componentAddArgument) + 1;
                boolean canAddArg = JavaRewriterUtil.constructorArgumentIsArray(objectCreationExpr, addIndex);
                if (canAddArg) {
                    expressionToAdd = objectCreationExpr;
                } else {
                    if (!JavaRewriterUtil.isConstructorArgumentFlowComponentType(objectCreationExpr, addIndex)) {
                        throw new IllegalArgumentException(
                                "Could not duplicate in constructor argument of " + objectCreationExpr);
                    }
                    ObjectCreationExpr divCreationExpression = JavaRewriterUtil.getWrapperDivObjectCreationExpr(
                            component.getAttachLocationCompilationUnitOrThrowIfNull(),
                            objectCreationExpr.getArguments());

                    List<Expression> arguments = objectCreationExpr.getArguments().stream().toList();
                    for (Expression argument : arguments) {
                        argument.remove();
                    }
                    // clear arguments
                    objectCreationExpr.addArgument(divCreationExpression);

                    expressionToAdd = divCreationExpression;
                    addIndex = 0;
                }
            } else {
                addIndex = component.getAttachCallInSameFileOrThrow().getNodeWithArguments()
                        .getArgumentPosition(componentAddArgument) + 1;
                expressionToAdd = component.getAttachCallInSameFileOrThrow().getNodeWithArguments();
            }
            if (duplicatedName != null) {
                expressionToAdd.getArguments().add(addIndex, new NameExpr(duplicatedName));
            } else {
                expressionToAdd.getArguments().add(addIndex,
                        JavaRewriterUtil.clone(component.getCreateInfoOrThrow().getObjectCreationExpr()));
            }
        }

        Map<String, String> nameMappings;
        if (oldName != null) {
            nameMappings = Collections.singletonMap(oldName, duplicatedName);
        } else {
            nameMappings = Collections.emptyMap();
        }

        return new DuplicateInfo(nameMappings, childAddCalls, variableDeclaration, assignExpr);
    }

    /**
     * Adds the given code snippet to the source code either before the reference
     * component (Where.BEFORE) or by appending to the layout (Where.APPEND).
     *
     * @param referenceComponent
     *            the reference component (BEFORE) or container (APPEND) to add the
     *            code
     * @param where
     *            where to add the code
     * @param template
     *            the code to add, as JSON array of objects with "tag", "props" and
     *            "children"
     * @param options
     *            options that control how the template is added
     * @return the created components
     */
    public List<ComponentInfo> addComponentUsingTemplate(ComponentInfo referenceComponent, Where where,
            List<JavaComponent> template, AddTemplateOptions options) {
        if (where == null && options.methodName() != null) {

            // where null means that user selected a method from drop method API dialog.
            // Thus,
            // given template should be added before reference component by default.
            // When route view is selected, Append is used
            where = Where.BEFORE;
            if (referenceComponent.routeConstructor() != null) {
                where = Where.APPEND;
            }
            return addComponentUsingTemplate(referenceComponent, where, template, options);
        }
        InsertionPoint insertionPoint = findInsertionPointForAppend(referenceComponent, where);
        String layoutVariableName = JavaRewriterUtil.getVariableNameForAdd(referenceComponent);
        JavaSource javaSource = referenceComponent.getCreateInfoOrThrow().getJavaSource();

        if (where == Where.APPEND) {
            //
            // HorizontalLayout layout = new HorizontalLayout(button);
            // layout.setThis();
            // layout.addThat();
            // ...
            // add(layout);
            if (layoutVariableName != null) {
                // we know what name to use for method calls
                return createComponentStatements(insertionPoint, null, template, layoutVariableName, null, options,
                        where, javaSource);
            } else {
                // we don't know what name to use for method calls so we include the anonymous
                // container
                return createComponentStatements(insertionPoint, null, template, null, referenceComponent, options,
                        where, javaSource);
            }
        } else if (where == Where.BEFORE) {
            // HorizontalLayout hl = new HorizontalLayout();
            // button = new Button("Hello");
            // button.setThis();
            // button.addThat();
            // hl.add(button);
            //
            // Insert constructor + setters before the reference constructor and
            // add the component so that it is added right before the reference
            return createComponentStatements(insertionPoint, null, template,
                    JavaRewriterUtil.getFieldOrVariableName(referenceComponent), referenceComponent, options, where,
                    javaSource);
        } else {
            throw new IllegalArgumentException(UNKNOWN_WHERE + where);
        }
    }

    private InsertionPoint findInsertionPointForAppend(ComponentInfo component) {
        List<MethodCallExpr> functionCalls = JavaRewriterUtil.findMethodCallStatements(component);
        if (!functionCalls.isEmpty()) {
            // There are some component.setThis() or component.somethingElse()
            // calls, add a new component after those
            MethodCallExpr lastCall = functionCalls.get(functionCalls.size() - 1);
            return JavaRewriterUtil.findLocationAfter(lastCall);
        } else if (component.routeConstructor() != null) {
            return JavaRewriterUtil.findLocationAtEnd(component.routeConstructor().getBody());
        }

        // There are no component.anything() calls, add before the
        // add(component) call
        return JavaRewriterUtil.findLocationBefore(component.getAttachCallInSameFileOrThrow().expression());
    }

    public InsertionPoint findInsertionPointForAppend(ComponentInfo component, Where where) {
        if (where == Where.APPEND) {
            return findInsertionPointForAppend(component);
        } else if (where == Where.BEFORE) {
            BlockStmt insertBlock;
            int insertIndex;

            if (component.getCreateInfoOrThrow().getFieldDeclarationAndAssignment() == null) {
                // component is instantiated a block ,so block that object creation present is
                // where new statements will be added
                insertBlock = component.getCreateInfoOrThrow().getComponentCreateScope();
                insertIndex = JavaRewriterUtil
                        .findBlockStatementIndex(component.getCreateInfoOrThrow().getObjectCreationExpr());
            } else {
                // component is instantiated in class as field variable, so block that attach
                // statement present is where new statements will be added
                AttachExpression attachExpression = component.getAttachCallInSameFileOrThrow();
                insertIndex = JavaRewriterUtil.findBlockStatementIndex(attachExpression.getNode());
                insertBlock = JavaRewriterUtil.findAncestorOrThrow(attachExpression.getNode(), BlockStmt.class);
            }
            return new InsertionPoint(insertBlock, insertIndex);

        } else {
            throw new IllegalArgumentException(UNKNOWN_WHERE + where);

        }
    }

    public void setAlignment(ComponentInfo component, String alignItemsClassName, String justifyContentClassName) {
        LumoRewriterUtil.removeClassNameArgs(component, "AlignItems", "JustifyContent");
        LumoRewriterUtil.addLumoUtilityImport(component.getCreateLocationCompilationUnitOrThrowIfNull());
        if (alignItemsClassName != null) {
            addOrReplaceLumoClass(component, "AlignItems", alignItemsClassName);
        }
        if (justifyContentClassName != null) {
            addOrReplaceLumoClass(component, "JustifyContent", justifyContentClassName);
        }
    }

    /**
     * Sets gap to selected component
     *
     * @param component
     *            component to set gap
     * @param lumoClassAll
     *            gap all value. e.g. gap-m. might be <code>null</code>
     * @param lumoClassColumn
     *            gap column value which starts with gap-x prefix, e.g. gap-x-xs.
     *            might be <code>null</code>
     * @param lumoClassRow
     *            gap row value which starts with gap-y prefix, e.g. gap-y-xs. might
     *            be <code>null</code>
     */
    public void setGap(ComponentInfo component, String lumoClassAll, String lumoClassColumn, String lumoClassRow) {
        LumoRewriterUtil.removeClassNameArgs(component, "Gap", "Gap.Column", "Gap.Row");
        LumoRewriterUtil.addLumoUtilityImport(component.getCreateLocationCompilationUnitOrThrowIfNull());
        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil.findMethodCallStatements(component);
        methodCallStatements.stream()
                .filter(methodCallExpr -> StringUtils.equals(methodCallExpr.getNameAsString(), SET_SPACING_METHOD))
                .findAny().ifPresent(JavaRewriterUtil::removeStatement);
        LumoRewriterUtil.removeThemeArgStartsWith(methodCallStatements, "spacing");

        if (lumoClassAll != null) {
            addOrReplaceLumoClass(component, "Gap", lumoClassAll);
        }
        if (lumoClassRow != null) {
            addOrReplaceLumoClass(component, "Gap.Row", lumoClassRow);
        }
        if (lumoClassColumn != null) {
            addOrReplaceLumoClass(component, "Gap.Column", lumoClassColumn);
        }
        if (lumoClassAll == null && lumoClassColumn == null && lumoClassRow == null) {
            MethodCallExpr call = JavaRewriterUtil.addAfterLastFunctionCall(methodCallStatements, SET_SPACING_METHOD,
                    new BooleanLiteralExpr(false));
            if (call == null) {
                replaceOrAddCall(component, SET_SPACING_METHOD, false);
            }
        }
    }

    private void addOrReplaceLumoClass(ComponentInfo component, String lumoUtilityClassName, String cssClassName) {
        List<Expression> gapExpressions = LumoRewriterUtil.getLumoMethodArgExpressions(List.of(lumoUtilityClassName),
                List.of(cssClassName));
        boolean added = LumoRewriterUtil.addClassNameWithArgs(component, gapExpressions);
        if (!added) {
            replaceOrAddCall(component, ADD_CLASS_NAMES_METHOD, gapExpressions.toArray(new Object[0]));
        }
    }

    public void setPadding(ComponentInfo component, String all, String top, String right, String bottom, String left) {
        // removing all padding classes
        String[] lumoPaddingInnerClassNames = new String[] { "Padding", "Padding.Bottom", "Padding.End",
                "Padding.Horizontal", "Padding.Left", "Padding.Right", "Padding.Start", "Padding.Top",
                "Padding.Vertical" };

        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil.findMethodCallStatements(component);
        methodCallStatements.stream()
                .filter(methodCallExpr -> StringUtils.equals(methodCallExpr.getNameAsString(), "setPadding"))
                .forEach(JavaRewriterUtil::removeStatement);
        LumoRewriterUtil.removeSetThemeArgs(methodCallStatements, "padding");

        LumoRewriterUtil.removeClassNameArgs(component, lumoPaddingInnerClassNames);
        LumoRewriterUtil.addLumoUtilityImport(component.getCreateLocationCompilationUnitOrThrowIfNull());

        if (all != null) {
            addOrReplaceLumoClass(component, "Padding", all);
        }
        if (top != null) {
            addOrReplaceLumoClass(component, "Padding.Top", top);
        }
        if (right != null) {
            addOrReplaceLumoClass(component, "Padding.Right", right);
        }
        if (bottom != null) {
            addOrReplaceLumoClass(component, "Padding.Bottom", bottom);
        }
        if (left != null) {
            addOrReplaceLumoClass(component, "Padding.Left", left);
        }
    }

    /**
     * Merges all the components and wraps them using the given component and places
     * the result in place of the first component.
     *
     * @param components
     *            The components to merge. The first component will be replaced with
     *            the wrapper component
     * @param wrapperComponent
     *            The component to wrap the merged components in.
     */
    public void mergeAndReplace(List<ComponentInfo> components, JavaComponent wrapperComponent) {
        // Goal is to add a wrapper before the first selected component and move all
        // selected components into that
        Where where = Where.BEFORE;
        ComponentInfo firstComponent = components.get(0);

        ComponentInfo wrapperComponentInfo = addComponentUsingTemplate(firstComponent, where, List.of(wrapperComponent),
                new AddTemplateOptions(false)).get(0);
        for (ComponentInfo component : components) {
            moveComponent(component, wrapperComponentInfo, null, Where.APPEND, null);
        }
    }

    /**
     * Extracts the given component, including its children, to a new component.
     *
     * @param component
     *            The component to extract.
     * @param componentChildren
     *            Children info for the component and its contents. This is an
     *            identity hash map to ensure that the component info is found even
     *            after the AST has been modified. A record in Java uses the data
     *            for hashcode/equals so they can change when e.g. a block changes,
     *            which would make a lookup in a normal hash map to fail.
     * @param extractedComponentClassNameSuggestion
     *            A suggestion for the name of the new component. Will be modified
     *            if it is already in use.
     */
    public void extractComponent(ComponentInfo component,
            IdentityHashMap<ComponentInfo, List<ComponentInfo>> componentChildren,
            String extractedComponentClassNameSuggestion) {
        if (component.routeConstructor() != null) {
            throw new CopilotException("Extracting component that has an own class is not possible.");
        }
        CompilationUnit compilationUnit = component.getCreateLocationCompilationUnitOrThrowIfNull();
        ClassOrInterfaceDeclaration classOrInterfaceDeclaration = JavaRewriterUtil.findOutermostAncestorOrThrow(
                component.getCreateInfoOrThrow().getObjectCreationExpr(), ClassOrInterfaceDeclaration.class);

        String extractedComponentClassName = JavaRewriterUtil
                .findFreeVariableName(extractedComponentClassNameSuggestion, classOrInterfaceDeclaration);
        // We need to figure out some configuration before we start modifying the AST
        String rootComponentVariableName = JavaRewriterUtil.getFieldOrVariableName(component);
        ClassOrInterfaceType extractedComponentType = StaticJavaParser
                .parseClassOrInterfaceType(extractedComponentClassName);
        Optional<Constructor<?>> maybeComponentConstructor = JavaRewriterUtil.findConstructor(component.type(),
                component.getCreateInfoOrThrow().getObjectCreationExpr());

        // Create the new component
        ClassOrInterfaceDeclaration newComponent = new ClassOrInterfaceDeclaration(
                NodeList.nodeList(Modifier.publicModifier(), Modifier.staticModifier()), false,
                extractedComponentClassName);
        JavaRewriterUtil.addImport(compilationUnit, component.type().getName());
        JavaRewriterUtil.addImport(compilationUnit, Composite.class.getName());
        newComponent.addExtendedType(
                StaticJavaParser.parseClassOrInterfaceType("Composite<" + component.typeWithGenerics() + ">"));
        ConstructorDeclaration extractedComponentConstructor = newComponent.addConstructor(Modifier.Keyword.PUBLIC);
        classOrInterfaceDeclaration.getMembers().add(0, newComponent);

        BlockStmt compositeConstructorBody = extractedComponentConstructor.getBody();

        String extractedComponentVariableName = JavaRewriterUtil.findFreeVariableName(
                SharedUtil.firstToLower(extractedComponentClassName),
                component.getComponentAttachScopeOrThrowIfDifferentFile().orElseThrow(
                        () -> new IllegalArgumentException("No attach scope found for the root component")));

        ObjectCreationExpr extractedComponentInitializer = new ObjectCreationExpr(null, extractedComponentType,
                NodeList.nodeList());
        // First create the root component as local variable or field, like it was
        // e.g. NativeButton button = getContent();
        MethodCallExpr getContentCall = new MethodCallExpr("getContent");
        var createInfo = component.getCreateInfoOrThrow();
        if (createInfo.getFieldDeclaration() != null) {
            ClassOrInterfaceDeclaration viewClass = JavaRewriterUtil
                    .findAncestorOrThrow(createInfo.getFieldDeclaration(), ClassOrInterfaceDeclaration.class);
            int fieldIndex = JavaRewriterUtil.findDeclarationIndex(createInfo.getFieldDeclaration());

            // Component was a field, so we create the field
            // private NativeButton button;
            // and the assignment using getContent()
            // button = getContent();

            VariableDeclarator rootComponentVariable = createInfo.getFieldDeclaration().getVariable(0);
            VariableDeclarator rootComponentVariableDeclarator = new VariableDeclarator(rootComponentVariable.getType(),
                    rootComponentVariableName);
            FieldDeclaration rootComponentFieldDeclaration = new FieldDeclaration(
                    NodeList.nodeList(Modifier.privateModifier()), rootComponentVariableDeclarator);
            newComponent.getMembers().add(0, rootComponentFieldDeclaration);

            FieldAccessExpr fieldAccessExpr = new FieldAccessExpr(new ThisExpr(), rootComponentVariableName);
            AssignExpr rootComponentAssignmentExpression = new AssignExpr(fieldAccessExpr, getContentCall,
                    AssignExpr.Operator.ASSIGN);
            compositeConstructorBody.addStatement(rootComponentAssignmentExpression);

            // Replace the original field with the extracted component
            VariableDeclarator extractedComponentInstanceVariable = new VariableDeclarator(extractedComponentType,
                    extractedComponentVariableName);
            FieldDeclaration extractedComponentInstanceField = new FieldDeclaration(
                    NodeList.nodeList(createInfo.getFieldDeclaration().getModifiers()),
                    extractedComponentInstanceVariable);
            if (createInfo.getFieldDeclarationAndAssignment() != null) {
                extractedComponentInstanceVariable.setInitializer(extractedComponentInitializer);
            } else {
                // Replace the original local variable with the extracted component
                ExpressionStmt newStatement = new ExpressionStmt(
                        new AssignExpr(new FieldAccessExpr(new ThisExpr(), extractedComponentVariableName),
                                extractedComponentInitializer, AssignExpr.Operator.ASSIGN));
                JavaRewriterUtil.findLocationBefore(component.getCreateInfoOrThrow().getObjectCreationExpr())
                        .add(newStatement);
            }
            // This is a workaround as replacing the field does not work without hacky
            // Observer/Merger..
            viewClass.getMembers().add(fieldIndex, extractedComponentInstanceField);
        } else {
            Type rootComponentType;
            if (createInfo.getLocalVariableDeclarator() != null) {
                VariableDeclarator rootComponentLocalVariableDeclarator = createInfo.getLocalVariableDeclarator();
                rootComponentType = rootComponentLocalVariableDeclarator.getType();
            } else {
                // Inline root component
                rootComponentType = component.getCreateInfoOrThrow().getObjectCreationExpr().getType();
                rootComponentVariableName = SharedUtil.firstToLower(rootComponentType.asString());
            }
            compositeConstructorBody.addStatement(new VariableDeclarationExpr(
                    new VariableDeclarator(rootComponentType, rootComponentVariableName, getContentCall)));
            // Replace the original local variable with the extracted component
            VariableDeclarator extractedComponentInstanceVariable = new VariableDeclarator(extractedComponentType,
                    extractedComponentVariableName);
            extractedComponentInstanceVariable.setInitializer(extractedComponentInitializer);
            ExpressionStmt newStatement = new ExpressionStmt(
                    new VariableDeclarationExpr(extractedComponentInstanceVariable));

            JavaRewriterUtil.findLocationBefore(component.getCreateInfoOrThrow().getObjectCreationExpr())
                    .add(newStatement);
        }

        // If the object creation expression had parameters, try to use the setters
        // instead
        if (maybeComponentConstructor.isPresent()) {
            Constructor<?> componentConstructor = maybeComponentConstructor.get();
            long parameterCount = Arrays.stream(componentConstructor.getParameters())
                    .filter(p -> !JavaRewriterUtil.isComponentOrComponentArray(p.getType())).count();
            if (parameterCount > 0) {
                Map<Integer, String> mappings = ConstructorAnalyzer.get().getMappings(componentConstructor);
                if (mappings.size() != parameterCount) {
                    // This should use some other pattern that creates the component manually
                    // instead
                    throw new IllegalArgumentException("Could not find mappings for all constructor parameters");
                }
                for (Map.Entry<Integer, String> entry : mappings.entrySet()) {
                    int parameterIndex = entry.getKey();
                    String setterName = entry.getValue();
                    Expression parameter = component.getCreateInfoOrThrow().getObjectCreationExpr()
                            .getArgument(parameterIndex);
                    compositeConstructorBody.addStatement(new MethodCallExpr(new NameExpr(rootComponentVariableName),
                            setterName, NodeList.nodeList(parameter)));
                }
            }
        }

        // Replace attach with the extracted component
        JavaRewriterUtil.getAttachArgument(component).ifPresent(attachArgument -> {
            attachArgument.replace(new NameExpr(extractedComponentVariableName));
        });

        InsertionPoint insertLocation = new InsertionPoint(compositeConstructorBody,
                compositeConstructorBody.getStatements().size());

        // Setters for the extracted component
        copyFunctionCalls(component, insertLocation);
        List<ComponentInfo> children = componentChildren.get(component);
        moveComponents(children, componentChildren, insertLocation, rootComponentVariableName,
                field -> newComponent.getMembers().add(newComponent.getMembers().size() - 1, field));

        delete(component);

        // Find names in the new component (constructor) that cannot be resolved and add
        // them as parameters
        List<Resolvable> nodes = newComponent.findAll(Node.class).stream().filter(node -> node instanceof Resolvable)
                .map(Resolvable.class::cast).toList();
        LinkedHashSet<String> unresolved = new LinkedHashSet<>();
        for (Resolvable<?> node : nodes) {
            try {
                ResolvedDeclaration resolved = (ResolvedDeclaration) node.resolve();
                // JavaParser incorrectly resolves fields outside the static class
                if (resolved.isField()) {
                    Optional<Node> ast = resolved.asField().toAst();
                    if (ast.isPresent() && JavaRewriterUtil.findAncestorOrThrow(ast.get(),
                            ClassOrInterfaceDeclaration.class) == classOrInterfaceDeclaration) {
                        unresolved.add(resolved.getName());
                    }
                }
            } catch (UnsolvedSymbolException e) {
                // JavaRewriter messes up context and name sometimes...
                String name = JavaRewriterUtil.getSymbolNameFromJavaParserException(e);
                unresolved.add(name);
                getLogger().debug("Could not resolve name {} in {}. Potentially needs to be a parameter", name, node,
                        e);
            } catch (Exception e) {
                getLogger().debug("Error resolving {}. Ignoring", node);
            }
        }

        for (String constructorParameter : unresolved) {
            NameExpr nameExpr = new NameExpr(constructorParameter);
            extractedComponentInitializer.addArgument(nameExpr);
            try {
                ResolvedValueDeclaration resolved = nameExpr.resolve();
                if (resolved.getType().isReferenceType()) {
                    extractedComponentConstructor.addParameter(
                            Util.getSimpleName(resolved.getType().asReferenceType().getQualifiedName()),
                            constructorParameter);
                }
            } catch (UnsolvedSymbolException e) {
                // Could not resolve the variable in the original scope. Should not happen...
                // but if it does, we don't know the type to use in the constructor so we skip
                // the parameter
                getLogger().warn("Unable to resolve type of {} in the original method", constructorParameter, e);
                extractedComponentInitializer.remove(nameExpr);
            }
        }

    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

    private void moveComponents(List<ComponentInfo> components,
            Map<ComponentInfo, List<ComponentInfo>> componentChildren, InsertionPoint insertLocation,
            String containerVariableName, Consumer<FieldDeclaration> addField) {
        if (components == null) {
            return;
        }
        List<ComponentInfo> toDelete = new ArrayList<>();
        components.forEach(component -> {
            // Create expression
            var createInfo = component.getCreateInfoOrThrow();
            VariableDeclarator childVariable = createInfo.getFieldDeclaration() != null
                    ? createInfo.getFieldDeclaration().getVariable(0)
                    : createInfo.getLocalVariableDeclarator();
            String childVariableName;
            Type childType;
            ObjectCreationExpr componentCreationExpr = component.getCreateInfoOrThrow().getObjectCreationExpr();
            if (childVariable != null) {
                childVariableName = childVariable.getNameAsString();
                childType = childVariable.getType();
            } else {
                childVariableName = JavaRewriterUtil.findFreeVariableName(
                        SharedUtil.firstToLower(component.type().getSimpleName()), insertLocation.getBlock());
                childType = componentCreationExpr.getType();
            }
            VariableDeclarator newVariable = new VariableDeclarator(childType,
                    JavaRewriterUtil.findFreeVariableName(childVariableName, insertLocation.getBlock()));
            List<ComponentInfo> childComponents = componentChildren.get(component);
            if (childComponents != null) {
                for (ComponentInfo childComponent : childComponents) {
                    AttachExpression attachExpression = childComponent.getAttachCallInSameFileOrThrow();
                    if (attachExpression != null
                            && attachExpression.getObjectCreationExpression() == componentCreationExpr) {
                        // We must remove children attached in the constructor - they are always
                        // attached using add
                        JavaRewriterUtil.getAttachArgument(childComponent).ifPresent(attachArgument -> {
                            attachArgument.remove();
                        });
                    }
                }
            }
            ObjectCreationExpr childCreateExpression = JavaRewriterUtil.clone(componentCreationExpr);
            if (createInfo.getFieldDeclaration() != null) {
                // Field
                FieldDeclaration field = new FieldDeclaration(NodeList.nodeList(Modifier.privateModifier()),
                        newVariable);
                addField.accept(field);
                insertLocation
                        .add(new ExpressionStmt(new AssignExpr(new FieldAccessExpr(new ThisExpr(), childVariableName),
                                childCreateExpression, AssignExpr.Operator.ASSIGN)));
            } else {
                // Local variable
                newVariable.setInitializer(childCreateExpression);
                insertLocation.add(new ExpressionStmt(new VariableDeclarationExpr(newVariable)));
            }

            // Setters
            copyFunctionCalls(component, insertLocation);

            // Attach
            insertLocation.add(new ExpressionStmt(new MethodCallExpr(new NameExpr(containerVariableName), "add",
                    NodeList.nodeList(new NameExpr(newVariable.getNameAsString())))));

            toDelete.add(component);

            moveComponents(childComponents, componentChildren, insertLocation, newVariable.getNameAsString(), addField);
        });

        toDelete.forEach(this::delete);
    }

    private void copyFunctionCalls(ComponentInfo component, InsertionPoint insertLocation) {
        List<MethodCallExpr> childFunctionCalls = JavaRewriterUtil.findMethodCallStatements(component);
        childFunctionCalls.stream().filter(methodCallExpr -> !JavaRewriterUtil.isAttachCall(methodCallExpr)).forEach(
                methodCallExpr -> insertLocation.add(new ExpressionStmt(JavaRewriterUtil.clone(methodCallExpr))));
    }

    /**
     * Replaces a parameter name in a method call.
     *
     * @param call
     * @param oldVariableName
     * @param newVariableName
     */
    public void replaceCallParameter(NodeWithArguments<?> call, String oldVariableName, String newVariableName) {
        call.getArguments().forEach(arg -> {
            if (arg.isNameExpr() && arg.asNameExpr().getNameAsString().equals(oldVariableName)) {
                arg.asNameExpr().setName(newVariableName);
            }
        });
    }

    /**
     * Imports the given components that come from a lit template into the given
     * LitTemplate Java class.
     *
     * @param litTemplateSource
     *            the source of the LitTemplate class
     * @param componentDefinitions
     *            component definitions extracted from the corresponding lit
     *            template typescript file
     */
    public void importLitTemplate(JavaSource litTemplateSource, List<JavaComponent> componentDefinitions) {
        // Get the compilation unit from the source
        CompilationUnit compilationUnit = litTemplateSource.getCompilationUnit();

        // Find the main class in the compilation unit
        Optional<ClassOrInterfaceDeclaration> classOpt = compilationUnit.findFirst(ClassOrInterfaceDeclaration.class);
        if (classOpt.isEmpty()) {
            return;
        }

        ClassOrInterfaceDeclaration classDecl = classOpt.get();

        // Remove all JsModule annotations
        classDecl.getAnnotations().removeIf(annotation -> annotation.getNameAsString().equals("JsModule"));

        // Change superclass from LitTemplate to Div
        classDecl.getExtendedTypes().clear();
        classDecl.addExtendedType("Div");
        JavaRewriterUtil.addImport(compilationUnit, "com.vaadin.flow.component.html.Div");

        // Find or create a constructor
        ConstructorDeclaration constructor = null;
        List<ConstructorDeclaration> constructors = classDecl.getConstructors();

        if (constructors.isEmpty()) {
            // Create a new constructor if none exists
            constructor = classDecl.addConstructor(Modifier.Keyword.PUBLIC);

            // Add a super() call if this is a subclass
            if (!classDecl.getExtendedTypes().isEmpty()) {
                BlockStmt body = constructor.getBody();
                body.addStatement(new ExplicitConstructorInvocationStmt(false, null, NodeList.nodeList()));
            }
        } else {
            // Use the first constructor
            constructor = constructors.get(0);
        }

        // Create an insertion point after the super call (if any)
        BlockStmt constructorBody = constructor.getBody();
        int insertIndex = 0;

        // Find the super() call if it exists
        Optional<ExplicitConstructorInvocationStmt> superCallOpt = constructorBody
                .findFirst(ExplicitConstructorInvocationStmt.class);
        if (superCallOpt.isPresent()) {
            // Insert after the super call
            insertIndex = constructorBody.getStatements().indexOf(superCallOpt.get()) + 1;
        }

        // Create the constructUI method
        String constructUiMethodName = JavaRewriterUtil.findFreeVariableName("constructUI", classDecl);
        MethodDeclaration constructUIMethod = classDecl.addMethod(constructUiMethodName, Modifier.Keyword.PRIVATE);
        BlockStmt constructUIMethodBody = new BlockStmt();
        constructUIMethod.setBody(constructUIMethodBody);

        constructorBody.addStatement(insertIndex, new MethodCallExpr(constructUiMethodName));

        // Create an insertion point for adding component statements to the constructUI
        // method
        InsertionPoint insertionPoint = new InsertionPoint(constructUIMethodBody, 0);

        createComponentStatements(insertionPoint, null, componentDefinitions, "this", null,
                new AddTemplateOptions(false, true, null, null), Where.BEFORE, litTemplateSource);
    }

    public List<ComponentInfo> createComponentStatements(InsertionPoint insertionPoint, JavaComponent parent,
            List<JavaComponent> template, String layoutVariableName, ComponentInfo referenceComponent,
            AddTemplateOptions options, Where where, JavaSource javaSource) {
        return template.stream().map(javaComponent -> {
            Optional<CustomComponentHandle> customComponentHandleOpt = CustomComponentHandler.get(javaComponent);
            if (customComponentHandleOpt.isPresent()) {
                customComponentHandleOpt.get().createComponentStatements(this, javaComponent, insertionPoint, parent,
                        layoutVariableName, referenceComponent, options, javaSource);
                return null;
            } else {
                return createComponentStatements(insertionPoint, parent, javaComponent, true, layoutVariableName,
                        referenceComponent, options, where, javaSource);
            }
        }).toList();
    }

    public ComponentInfo createComponentStatements(InsertionPoint insertionPoint, JavaComponent parent,
            JavaComponent maybeJavaComponent, boolean attach, String layoutVariableName,
            ComponentInfo referenceComponent, AddTemplateOptions options, Where where, JavaSource javaSource) {

        final JavaComponent javaComponent;

        if ("Message".equals(maybeJavaComponent.tag()) || "Item".equals(maybeJavaComponent.tag())
                || "RadioButton".equals(maybeJavaComponent.tag()) || "CustomField".equals(maybeJavaComponent.tag())) {
            // A custom field is not supported right now as it needs a custom
            // class
            // vaadin-message has no Flow component
            // vaadin-radio-button has no Flow component
            // VaadinItem cannot be used without listbox
            return null;
        }
        String componentClassName = maybeJavaComponent.className() != null ? maybeJavaComponent.className()
                : FlowComponentQuirks.getClassForComponent(maybeJavaComponent);
        if (componentClassName.contains("$")) {
            componentClassName = componentClassName.replace("$", ".");
        }
        if (componentClassName.equals("com.vaadin.flow.component.treegrid.TreeGrid")) {
            // React does not differ between Grid and TreeGrid
            javaComponent = maybeJavaComponent.withTag("TreeGrid");
        } else {
            javaComponent = maybeJavaComponent;
        }

        ClassOrInterfaceType fullType = StaticJavaParser.parseClassOrInterfaceType(componentClassName);

        Map<String, Object> setters = new HashMap<>();
        Class<?> componentType = JavaReflectionUtil.getClass(fullType.getNameWithScope());

        // Import
        CompilationUnit compilationUnit = insertionPoint.getCompilationUnit();
        JavaRewriterUtil.addImport(compilationUnit, fullType.getNameWithScope());
        ClassOrInterfaceType variableType = JavaRewriterUtil.clone(fullType).removeScope();
        ClassOrInterfaceType itemsType = null;
        String itemsTypeName = javaComponent.metadata().getItemType();

        if (itemsTypeName != null) {
            itemsType = JavaRewriterUtil.clone(StaticJavaParser.parseClassOrInterfaceType(itemsTypeName)).removeScope();
        }
        // we create a second type for instantiation to avoid explicit type and
        // allow to modify separately the ClassOrInterfaceType for declaration
        // and definition
        ClassOrInterfaceType typeForInstantiation = JavaRewriterUtil.clone(variableType);

        // Find data entity available name in case of data provider
        String dataEntityRecordName = JavaRewriterUtil.findFreeRecordName(DEFAULT_ENTITY_RECORD_NAME,
                JavaRewriterUtil.findAncestor(insertionPoint.getBlock(), ClassOrInterfaceDeclaration.class));

        // Constructor call
        ObjectCreationExpr initializer = JavaRewriterUtil.createComponentConstructor(javaComponent,
                typeForInstantiation, itemsType != null ? itemsType.getName().asString() : dataEntityRecordName,
                (String neededImport) -> JavaRewriterUtil.addImport(compilationUnit, neededImport));
        if (itemsType != null) {
            initializer.getType().setTypeArguments(JavaRewriterUtil.createEmptyType());
            JavaRewriterUtil.addImport(compilationUnit, itemsTypeName);
            variableType.setTypeArguments(itemsType);
        }
        // Gets a valid available variable name for the new component
        String variableName = JavaRewriterUtil.generateVariableName(javaComponent, variableType, insertionPoint);

        // Check if we should reuse an existing field with @Id annotation
        FieldDeclaration fieldToModify = null;

        if (options.useIdMappedFields()) {
            Object idProp = javaComponent.props().get("id");
            if (idProp instanceof String id && !id.isEmpty()) {
                fieldToModify = JavaRewriterUtil.findIdAnnotatedField(id, JavaRewriterUtil
                        .findAncestorOrThrow(insertionPoint.getBlock(), ClassOrInterfaceDeclaration.class));
            }
        }

        if (fieldToModify != null) {
            // Reuse the existing field
            variableName = fieldToModify.getVariables().get(0).getNameAsString();

            // Remove the @Id annotation
            fieldToModify.getAnnotations().removeIf(annotation -> annotation.getNameAsString().equals("Id"));

            // Add an initializer for the field
            insertionPoint.add(new ExpressionStmt(
                    new AssignExpr(new NameExpr(variableName), initializer, AssignExpr.Operator.ASSIGN)));
        } else if (options.javaFieldsForLeafComponents() && javaComponent.children().isEmpty()) {
            ClassOrInterfaceDeclaration classType = JavaRewriterUtil.findAncestorOrThrow(insertionPoint.getBlock(),
                    ClassOrInterfaceDeclaration.class);
            FieldDeclaration fieldDeclaration = classType.addFieldWithInitializer(variableType, variableName,
                    initializer, Modifier.Keyword.PRIVATE, Modifier.Keyword.FINAL);
            // Move field before all constructors/methods
            JavaRewriterUtil.moveAboveMethodsAndConstructors(fieldDeclaration, classType);
        } else {
            VariableDeclarator decl = new VariableDeclarator(variableType, variableName, initializer);
            VariableDeclarationExpr declarationExpr = new VariableDeclarationExpr(decl);
            insertionPoint.add(new ExpressionStmt(declarationExpr));
        }
        // A FieldAccessExpr always requires a scope, but we are verifying that the
        // field
        // name is unique, so we do not need that
        Expression variableReference = new NameExpr(variableName);
        JavaDataProviderHandler.FieldOrVariable fieldOrVariable = new JavaDataProviderHandler.FieldOrVariable(
                variableReference, variableType, initializer);

        // CHART HACK: THE CHART IS NOT VISIBLE BY DEFAULT BECAUSE OF ITS SIZE
        // SO WE SET A DEFAULT SIZE TO MAKE IT VISIBLE
        if (javaComponent.tag() != null && javaComponent.tag().equalsIgnoreCase("Chart")) {
            insertSetter(insertionPoint, fieldOrVariable.reference(), "setMinHeight", "400px", javaComponent,
                    javaSource);
        }

        // We add data items included as props when the item is set
        // with in a data provider bean like
        // ListDataView::setItems(T... items)
        // SO FIRST WE HANDLE PROPERTIES
        JavaDataProviderHandler.handleDataStatementsAndClearDataProps(compilationUnit, fieldOrVariable, javaComponent,
                insertionPoint, dataEntityRecordName);

        // Properties setters
        javaComponent.props().forEach((prop, value) -> {
            if (prop.startsWith("_")) {
                // These are used for passing parent props through a child, e.g. label for a tab
                // sheet tab content
                return;
            }
            if (FlowComponentQuirks.skipProps(javaComponent, prop)) {
                return;
            }
            SetterAndValue setterAndValue = JavaRewriterUtil.getSetterAndValue(componentType, prop, value);
            setters.put(setterAndValue.setter(), setterAndValue.value());
        });

        // Inner text setter
        Optional<String> innerText = javaComponent.innerText();
        if (innerText.isPresent()) {
            SetterAndValue setterAndValue = JavaRewriterUtil.getSetterAndValue(componentType,
                    FlowComponentQuirks.getInnerTextProperty(componentType), innerText.get());
            setters.put(setterAndValue.setter(), setterAndValue.value());
        }
        // Removes setter property to be used in constructor
        JavaRewriterUtil.getSingleStringParamConstructor(fullType, setters.keySet()).ifPresent(constructorProp -> {
            Object value = setters.remove(constructorProp);
            getParameterList(insertionPoint, value, javaSource).forEach(initializer::addArgument);
        });

        for (Map.Entry<String, Object> setter : setters.entrySet()) {
            Object value = setter.getValue();
            if (value instanceof JavaComponent javaComponentValue) {
                value = createSubComponentStatements(insertionPoint, javaComponent, List.of(javaComponentValue),
                        options, where, javaSource).get(0);
            } else if (value instanceof JavaComponent[] javaComponentValues) {
                value = createSubComponentStatements(insertionPoint, javaComponent, List.of(javaComponentValues),
                        options, where, javaSource);
            } else if (setter.getKey().equalsIgnoreCase(ADD_ITEM_METHOD) && value instanceof List<?> list) {
                for (Object item : list) {
                    insertItemsPropToAddItem(compilationUnit, javaComponent, insertionPoint,
                            fieldOrVariable.reference(), null, setter.getKey(), item);
                }
                continue;
            }
            insertSetter(insertionPoint, fieldOrVariable.reference(), setter.getKey(), value, javaComponent,
                    javaSource);
        }

        // Child components
        // First we remove children that are not meant to be added as components
        // but as method calls
        // THEN WE MANAGE CHILDREN
        List<JavaComponent> methodChildren = FlowComponentQuirks.getMethodCallChildren(javaComponent);

        javaComponent.children().removeAll(methodChildren);

        // Inner texts are added through a property
        javaComponent.children().removeIf(child -> child.tag().equals(JavaComponent.TEXT_NODE));

        // Then we add the children as method calls
        createMethodCallChildrenStatements(compilationUnit, insertionPoint, javaComponent, methodChildren,
                fieldOrVariable.reference(), dataEntityRecordName);

        // Then we add the children as components when items are not
        // added with a data provider but as dedicated types of items like
        // MessageList::setItems(com.vaadin.flow.component.messages.MessageListItem...
        // items )
        List<JavaComponent> children = javaComponent.children().stream().map(child -> {
            if (FlowComponentQuirks.isTabSheetDefinition(child)) {
                JavaComponent tabContent;
                if (child.children().isEmpty()) {
                    // Use an empty div as there must be some content
                    Map<String, Object> emptyText = new HashMap<>();
                    emptyText.put("text", "");
                    tabContent = new JavaComponent("div", null, emptyText, Collections.emptyList());
                } else {
                    tabContent = child.children().get(0);
                }
                tabContent.props().put("_label", child.props().get("label"));
                return tabContent;
            }
            return child;
        }).toList();
        createComponentStatements(insertionPoint, javaComponent, children, variableName, null, options, where,
                javaSource);

        AttachExpression attachExpression = null;
        if (attach) {
            attachExpression = attachComponent(insertionPoint, javaComponent, parent, layoutVariableName,
                    referenceComponent, fieldOrVariable.reference(), variableName, options, where);
        }

        return JavaRewriterUtil.createComponentInfoForTemplateComponent(javaSource, initializer, attachExpression);
    }

    public AttachExpression attachComponent(InsertionPoint insertionPoint, JavaComponent component,
            JavaComponent parent, String layoutVariableName, ComponentInfo referenceComponent,
            Expression variableNameExpr, String variableName, AddTemplateOptions options, Where where) {
        if (options.methodName() != null && referenceComponent != null) {
            Optional<MethodCallExpr> methodCallExprOptional = JavaRewriterUtil
                    .findMethodCallStatements(referenceComponent, options.methodName()).findFirst();
            MethodCallExpr methodCallExpr;
            methodCallExpr = methodCallExprOptional.orElseGet(() -> JavaRewriterUtil.addFunctionCall(referenceComponent,
                    options.methodName(), new ArrayList<>()));
            boolean arrayArgument = JavaReflectionUtil.isArrayArgument(referenceComponent.type().getName(),
                    methodCallExpr.getNameAsString(), 0);
            if (AddReplace.REPLACE == options.addReplace) {
                // removes all method args and appends the new one to given attach call method
                List<Expression> currentArgs = new ArrayList<>(methodCallExpr.getArguments().stream().toList());
                JavaRewriterUtil.removeArgumentCalls(methodCallExpr, currentArgs, false);
                methodCallExpr.getArguments().add(new NameExpr(variableName));
                return new AttachExpression(methodCallExpr);
            } else if (AddReplace.ADD == options.addReplace) {
                if (arrayArgument) {
                    // if method argument is array, appending the new one
                    methodCallExpr.getArguments().add(new NameExpr(variableName));
                } else {
                    // if method argument is not an array, then wrap them with Div
                    List<Expression> currentArgs = new ArrayList<>(methodCallExpr.getArguments().stream().toList());
                    JavaRewriterUtil.removeArgumentCalls(methodCallExpr, currentArgs, false);
                    Expression expressionToAddAsArgs;
                    if (!currentArgs.isEmpty()) {
                        CompilationUnit compilationUnit = referenceComponent
                                .getAttachLocationCompilationUnitOrThrowIfNull();
                        JavaRewriterUtil.addImport(compilationUnit, "com.vaadin.flow.component.html.Div");
                        ObjectCreationExpr wrapperDivCreationExpr = new ObjectCreationExpr();
                        wrapperDivCreationExpr.setType("Div");
                        currentArgs.forEach(wrapperDivCreationExpr::addArgument);
                        wrapperDivCreationExpr.addArgument(new NameExpr(variableName));
                        expressionToAddAsArgs = wrapperDivCreationExpr;
                    } else {
                        expressionToAddAsArgs = new NameExpr(variableName);
                    }
                    methodCallExpr.addArgument(expressionToAddAsArgs);
                }
                return new AttachExpression(methodCallExpr);
            } else {
                throw new IllegalArgumentException("Unsupported addReplace option");
            }
        }
        // Attach component
        if (component.tag() != null && component.tag().equalsIgnoreCase(CHART_SERIES_CLASS)) {
            // ChartSeries are added to the Chart, not to the layout
            MethodCallExpr getConfigurationCall = new MethodCallExpr(new NameExpr(layoutVariableName),
                    "getConfiguration", new NodeList<>());
            MethodCallExpr addSeriesCall = new MethodCallExpr(getConfigurationCall, "addSeries",
                    new NodeList<>(variableNameExpr));
            insertionPoint.add(new ExpressionStmt(addSeriesCall));
            return new AttachExpression(addSeriesCall);
        } else if (referenceComponent != null) {
            AttachExpression referenceAttach = referenceComponent.getAttachCallInSameFileOrThrow();
            // We need to do this hack because the AttachLocation from tracker is not
            // correct for inline containers. We look if ancestor is an object creation like
            // new Div(component)
            NodeWithArguments<?> container = null;
            try {
                container = (NodeWithArguments<?>) referenceComponent.getCreateInfoOrThrow().getObjectCreationExpr()
                        .getParentNode().orElseThrow();
            } catch (ClassCastException e) {
                getLogger().debug("Unable to find parent node for reference component", e);
            }
            if (referenceComponent.isAnonymousComponent() && where == Where.APPEND) {
                // We attach the component to an anonymous container, addFunctionCall will
                // extract the anonymous class to a new named instance.
                // If the component is appended the referenceComponent is the proper container
                // the add call would be added afterward and then the component would be
                // appended.
                MethodCallExpr addCall = JavaRewriterUtil.addFunctionCall(referenceComponent, "add",
                        new NodeList<>(variableNameExpr));
                return new AttachExpression(addCall);
            } else if (referenceComponent.isAnonymousComponent()
                    && container instanceof ObjectCreationExpr objectCreationExpr) {
                // If the component must be added BEFORE we need to check if it was using the
                // constructor
                // If the container was using add method we go through standard case
                objectCreationExpr.getArguments().add(objectCreationExpr.getArguments()
                        .indexOf(referenceComponent.getCreateInfoOrThrow().getObjectCreationExpr()), variableNameExpr);
                return new AttachExpression(objectCreationExpr);
            } else if (referenceAttach.getNodeWithArguments().getArguments().size() == 1) {
                // add(reference)
                ExpressionStmt addCall = createAttachCall(component, parent,
                        referenceAttach.getNodeWithOptionalScope().getScope()
                                .orElse(JavaRewriterUtil.isNodeInCompositeClass(referenceAttach.getNode())
                                        ? new MethodCallExpr("getContent")
                                        : null),
                        variableNameExpr, options.methodName);
                BlockStmt block = JavaRewriterUtil.findAncestorOrThrow(referenceAttach.getNode(), BlockStmt.class);
                block.addStatement(JavaRewriterUtil.findBlockStatementIndex(referenceAttach.getNode()), addCall);
                return new AttachExpression(addCall.getExpression());
            } else {
                Expression referenceArgument;
                // add(..., reference,...)
                if (layoutVariableName == null) {
                    // Reference is an inline call, like add(new Button()
                    referenceArgument = referenceComponent.getCreateInfoOrThrow().getObjectCreationExpr();
                } else {
                    referenceArgument = referenceAttach.getNodeWithArguments().getArguments().stream()
                            .filter(arg -> arg instanceof NameExpr nameExpr
                                    && nameExpr.getNameAsString().equals(layoutVariableName))
                            .findFirst().orElse(null);
                }
                if (referenceArgument == null) {
                    throw new IllegalArgumentException(
                            "Did not find reference argument ('" + variableName + "') in " + referenceAttach);
                }
                int index = referenceAttach.getNodeWithArguments().getArguments().indexOf(referenceArgument);
                referenceAttach.getNodeWithArguments().getArguments().add(index, variableNameExpr);
                return referenceAttach;
            }
        } else if (layoutVariableName != null) {
            Expression scope = null;
            if (!layoutVariableName.equals("this")) {
                scope = new NameExpr(layoutVariableName);
            } else if (JavaRewriterUtil.isNodeInCompositeClass(insertionPoint.getBlock())) {
                scope = new MethodCallExpr("getContent");
            }
            ExpressionStmt addCall = createAttachCall(component, parent, scope, variableNameExpr, options.methodName);
            // Just adding to the given layout
            insertionPoint.add(addCall);
            return new AttachExpression(addCall.getExpression());
        } else {
            throw new IllegalArgumentException(
                    "Either layoutVariableName or referenceAttach must be given to attach a component");
        }
    }

    public NodeList<Expression> getParameterList(InsertionPoint insertionPoint, Object value, JavaSource javaSource) {
        if (value instanceof List<?> list) {
            if (list.isEmpty()) {
                return new NodeList<>();
            }
            if (list.get(0) instanceof JavaComponent) {
                return JavaRewriterUtil.toExpressionList(createSubComponentStatements(insertionPoint, null,
                        (List<JavaComponent>) list, new AddTemplateOptions(false), null, javaSource));
            }
        }
        return JavaRewriterUtil.toExpressionList(value);
    }

    /**
     * Insert setters for given java component. TODO move this class into
     * {@link CustomComponentHandle} and remove it from here
     *
     * @param insertionPoint
     *            Insertion point to add setters
     * @param owner
     *            owner of the setters
     * @param setterName
     *            setterName from JavaComponent props
     * @param value
     *            setterValue from JavaComponent props
     * @param javaComponent
     *            Java component itself
     * @param javaSource
     */
    public void insertSetter(InsertionPoint insertionPoint, Expression owner, String setterName, Object value,
            JavaComponent javaComponent, JavaSource javaSource) {
        if (setterName.equals("setStyle")) {
            insertStyles(insertionPoint, owner, (Map<String, String>) value);
        } else if (setterName.equals("setClassName")) {
            // replace setClassName with addClassNames and use Lumo Variables if possible.
            Expression addClassNameExp = LumoRewriterUtil.createAddClassNameExprUsingLumoVariables(owner, value,
                    insertionPoint.getCompilationUnit());
            if (addClassNameExp == null) {
                NodeList<Expression> parameterExpression = getParameterList(insertionPoint, value, javaSource);
                addClassNameExp = new MethodCallExpr(owner, setterName, parameterExpression);
            }
            insertionPoint.add(new ExpressionStmt(addClassNameExp));
        } else if (setterName.equals("setSlot")) {
            // getElement().setAttribute("slot", value)
            MethodCallExpr getElement = new MethodCallExpr(owner, "getElement");
            MethodCallExpr setAttributeCall = new MethodCallExpr(getElement, "setAttribute",
                    new NodeList<>(JavaRewriterUtil.toExpression("slot"), JavaRewriterUtil.toExpression(value)));
            insertionPoint.add(new ExpressionStmt(setAttributeCall));
        } else if (javaComponent.tag() != null && javaComponent.tag().equalsIgnoreCase("Chart")
                && !setterName.equals("setMinHeight")) {
            if (setterName.equals("setAdditionalOptions")) {
                Map<String, Object> additionalOptions = (Map<String, Object>) value;
                for (Map.Entry<String, Object> entry : additionalOptions.entrySet()) {
                    String key = entry.getKey();
                    Object val = entry.getValue();
                    if (key.equalsIgnoreCase("xAxis") || key.equalsIgnoreCase("yAxis")) {
                        Map<String, Object> axis = (Map<String, Object>) val;
                        for (Map.Entry<String, Object> entryAxis : axis.entrySet()) {
                            String keyAxis = entryAxis.getKey();
                            Object valAxis = entryAxis.getValue();
                            if (keyAxis.equalsIgnoreCase("categories")) {
                                insertionPoint.add(new ExpressionStmt(FlowComponentQuirks.getPropertySetExpression(
                                        javaComponent, key, keyAxis, valAxis, owner, insertionPoint)));
                            } else if (keyAxis.equalsIgnoreCase("title")) {
                                insertionPoint.add(new ExpressionStmt(FlowComponentQuirks.getPropertySetExpression(
                                        javaComponent, key, keyAxis, valAxis, owner, insertionPoint)));
                            }
                        }
                    }
                }
            } else {
                NodeList<Expression> parameterExpression = getParameterList(insertionPoint, value, javaSource);
                MethodCallExpr setterCall = new MethodCallExpr(FlowComponentQuirks
                        .getPropertySetExpression(javaComponent, null, setterName, value, owner, insertionPoint),
                        setterName, parameterExpression);
                insertionPoint.add(new ExpressionStmt(setterCall));
            }
        } else if (javaComponent.tag() != null && javaComponent.tag().equalsIgnoreCase(CHART_SERIES_CLASS)
                && setterName.equals("setPlotOptions")) {
            insertionPoint.add(new ExpressionStmt(FlowComponentQuirks.getPropertySetExpression(javaComponent, null,
                    setterName, value, owner, insertionPoint)));
        } else {
            NodeList<Expression> parameterExpression = getParameterList(insertionPoint, value, javaSource);
            if (CHART_SERIES_CLASS.equals(javaComponent.tag()) && "setData".equals(setterName)
                    && value instanceof List<?> list && !(list).isEmpty() && (list).get(0) instanceof JavaComponent) {
                // We need to wrap to a List only if the data is DataSeriesItem
                // instances because
                // DataSeries has
                // - setData(Number... values)
                // - setData(List<DataSeriesItem> data)
                parameterExpression = wrapWithListOf(parameterExpression);
                JavaRewriterUtil.addImport(insertionPoint.getCompilationUnit(), "java.util.List");
            }
            MethodCallExpr setterCall = new MethodCallExpr(owner, setterName, parameterExpression);
            insertionPoint.add(new ExpressionStmt(setterCall));
            if (value instanceof Enum<?>) {
                JavaRewriterUtil.addImport(insertionPoint.getCompilationUnit(), value.getClass().getCanonicalName());
            }
        }
    }

    /**
     * Connects the given grid to the given service method, using the given item
     * type and order for the columns.
     *
     * @param grid
     *            the grid to connect
     * @param serviceClassName
     *            The service class to use as data source
     * @param serviceMethodName
     *            the service method to use as data source
     * @param parameterTypes
     *            the service method parameter types
     * @param itemTypeClassName
     *            the item type to show in the grid, must correspond to the service
     *            method return type argument
     * @param itemPropertiesOrder
     *            the order of the properties to show in the grid
     */
    public void setGridDataSource(ComponentInfo grid, String serviceClassName, String serviceMethodName,
            List<JavaReflectionUtil.ParameterTypeInfo> parameterTypes, String itemTypeClassName,
            List<UIServiceCreator.FieldInfo> itemPropertiesOrder) {

        String serviceVariableName = addServiceToConstructor(grid, serviceClassName);

        // new Grid<>(Something.class, whatever) -> new Grid<>(Product.class);
        ClassOrInterfaceType itemType = setVariableGenericType(grid, itemTypeClassName);
        ClassExpr classExpr = new ClassExpr(itemType);
        classExpr.getType().asClassOrInterfaceType().removeScope();
        grid.getCreateInfoOrThrow().getObjectCreationExpr().getArguments().stream().toList().forEach(Node::remove);
        grid.getCreateInfoOrThrow().getObjectCreationExpr().addArgument(classExpr);

        removeCalls(grid, "setDataProvider");
        removeCalls(grid, "addColumn");
        removeCalls(grid, "addColumns");
        removeCalls(grid, "setColumnOrder");

        String itemsCode, methodName;
        if (parameterTypes.isEmpty()) {
            // Eager binding
            itemsCode = "SERVICEVARIABLE.METHODNAME()";
            methodName = "setItems";
            removeCalls(grid, "setItemsPageable");
        } else {
            methodName = "setItemsPageable";
            removeCalls(grid, "setItems");
            if (parameterTypes.size() == 2) {
                // Hilla list service
                itemsCode = "pageable -> SERVICEVARIABLE.METHODNAME(pageable, null)";
            } else {
                itemsCode = "SERVICEVARIABLE::METHODNAME";
            }

        }
        itemsCode = itemsCode.replace("SERVICEVARIABLE", serviceVariableName).replace("METHODNAME", serviceMethodName);
        replaceOrAddCall(grid, methodName, new Code(itemsCode));

        replaceOrAddCall(grid, "setColumns", (Object[]) itemPropertiesOrder.stream()
                .map(UIServiceCreator.FieldInfo::name).map(StringLiteralExpr::new).toArray(Expression[]::new));
    }

    /**
     * Connects the given combo box to the given service method, using the given
     * item type and order for the columns.
     *
     * @param comboBox
     *            the combo box to connect
     * @param serviceClassName
     *            The service class to use as data source
     * @param serviceMethodName
     *            the service method to use as data source
     * @param parameterTypes
     *            the service method parameter types
     * @param itemTypeClassName
     *            the item type to show in the grid, must correspond to the service
     *            method return type argument
     * @param displayProperty
     *            the property to display in the combo box
     */
    public void setComboBoxDataSource(ComponentInfo comboBox, String serviceClassName, String serviceMethodName,
            List<JavaReflectionUtil.ParameterTypeInfo> parameterTypes, String itemTypeClassName,
            UIServiceCreator.FieldInfo displayProperty) {

        String serviceVariableName = addServiceToConstructor(comboBox, serviceClassName);
        setVariableGenericType(comboBox, itemTypeClassName);

        removeCalls(comboBox, "setDataProvider");

        String itemsCode, methodName;
        if (hasParameters(parameterTypes)) {
            // Eager binding, in memory filtering by combobox itself
            itemsCode = "SERVICEVARIABLE.METHODNAME()";
            methodName = "setItems";
            removeCalls(comboBox, "setItemsPageable");
        } else if (hasParameters(parameterTypes, PAGEABLE_TYPE, STRING_TYPE)) {
            itemsCode = "SERVICEVARIABLE::METHODNAME";
            methodName = "setItemsPageable";
            removeCalls(comboBox, "setItems");
        } else if (hasParameters(parameterTypes, STRING_TYPE, PAGEABLE_TYPE)) {
            itemsCode = "(pageable, filter) -> SERVICEVARIABLE.METHODNAME(filter, pageable)";
            methodName = "setItemsPageable";
            removeCalls(comboBox, "setItems");
        } else if (hasParameters(parameterTypes, PAGEABLE_TYPE, HILLA_FILTER_TYPE)) {
            itemsCode = """
                        (pageable, filterString) -> {
                        PropertyStringFilter filter = new PropertyStringFilter();
                        filter.setPropertyId(DISPLAY_PROPERTY);
                        filter.setMatcher(PropertyStringFilter.Matcher.CONTAINS);
                        filter.setFilterValue(filterString);
                        return SERVICEVARIABLE.METHODNAME(pageable, filter);
                    }
                    """;
            methodName = "setItemsPageable";
            removeCalls(comboBox, "setItems");
        } else {
            throw new IllegalArgumentException("Unsupported parameter types: " + parameterTypes);
        }
        String displayPropertyName = displayProperty == null ? "" : displayProperty.name();
        itemsCode = itemsCode.replace("SERVICEVARIABLE", serviceVariableName).replace("METHODNAME", serviceMethodName)
                .replace("DISPLAY_PROPERTY", "\"" + displayPropertyName + "\"");
        replaceOrAddCall(comboBox, methodName, new Code(itemsCode));

        if (displayProperty == null) {
            removeCalls(comboBox, "setItemLabelGenerator");
        } else {
            MethodReferenceExpr displayPropertyReference = new MethodReferenceExpr(
                    new NameExpr(Util.getSimpleName(itemTypeClassName)), null,
                    Util.getGetterName(displayProperty.name(), displayProperty.javaType()));
            replaceOrAddCall(comboBox, "setItemLabelGenerator", displayPropertyReference);
        }
    }

    private ClassOrInterfaceType setVariableGenericType(ComponentInfo component, String genericTypeName) {
        ClassOrInterfaceType itemType = StaticJavaParser.parseClassOrInterfaceType(Util.getSimpleName(genericTypeName));
        VariableDeclarator variableDeclarator = component.getVariableDeclarator();
        if (variableDeclarator == null) {
            throw new IllegalArgumentException("Unable to find variable declarator for " + component);
        }
        // Component<Something> theComponent -> Component<itemType> theComponent
        ClassOrInterfaceType classOrInterfaceType = variableDeclarator.getType().asClassOrInterfaceType();
        classOrInterfaceType.setTypeArguments(new NodeList<>(itemType));
        CompilationUnit compilationUnit = component.getCreateLocationCompilationUnitOrThrowIfNull();
        compilationUnit.addImport(genericTypeName.replace("$", "."));
        return itemType;
    }

    private boolean hasParameters(List<JavaReflectionUtil.ParameterTypeInfo> parameterTypes, String... expectedTypes) {
        if (parameterTypes.size() != expectedTypes.length) {
            return false;
        }

        return IntStream.range(0, parameterTypes.size())
                .allMatch(i -> parameterTypes.get(i).type().typeName().equals(expectedTypes[i]));
    }

    private NodeList<Expression> wrapWithListOf(NodeList<Expression> parameterExpression) {
        return new NodeList<>(new MethodCallExpr(new NameExpr("List"), "of", parameterExpression));
    }

    private void insertStyles(InsertionPoint insertionPoint, Expression owner, Map<String, String> styles) {
        // button.getStyle().set("p1","value1").set("p2","value2");
        MethodCallExpr finalCall = new MethodCallExpr(owner, "getStyle");
        for (Map.Entry<String, String> entry : styles.entrySet()) {
            finalCall = new MethodCallExpr(finalCall, "set", new NodeList<>(
                    JavaRewriterUtil.toExpression(entry.getKey()), JavaRewriterUtil.toExpression(entry.getValue())));
        }
        insertionPoint.add(new ExpressionStmt(finalCall));
    }

    private NodeList<Expression> createSubComponentStatements(InsertionPoint insertionPoint, JavaComponent parent,
            List<JavaComponent> components, AddTemplateOptions options, Where where, JavaSource javaSource) {
        List<ComponentInfo> createdComponents = components.stream()
                .map(javaComponent -> createComponentStatements(insertionPoint, parent, javaComponent, false, null,
                        null, options, where, javaSource))
                .toList();
        return new NodeList<>(createdComponents.stream().map(ComponentInfo::getVariableDeclarator)
                .map(VariableDeclarator::getName).map(NameExpr::new).toArray(NameExpr[]::new));
    }

    private ExpressionStmt createAttachCall(JavaComponent component, JavaComponent parent, Expression scope,
            Expression toAdd, String selectedMethodName) {
        String methodName = selectedMethodName == null ? "add" : selectedMethodName;
        NodeList<Expression> parameters = new NodeList<>(toAdd);
        if (parent != null) {
            if (parent.tag().equals("SideNav") || parent.tag().equals("SideNavItem")) {
                methodName = ADD_ITEM_METHOD;
            } else if (parent.tag().equals("TabSheet")) {
                parameters.add(0, new StringLiteralExpr((String) component.props().get("_label")));
            } else if (parent.tag().equals("DashboardWidget")) {
                methodName = "setContent";
            } else if ((parent.tag().equals("Dashboard") || parent.tag().equals("DashboardLayout"))
                    && component.tag().equals("DashboardSection")) {
                methodName = "addSection";
            }
        }
        return new ExpressionStmt(new MethodCallExpr(scope, methodName, parameters));
    }

    /**
     * This method is used to create method call statements for the children of a
     * component when the mapping from React template to Java code is <br>
     * React children -> Flow method calls.<br>
     * This mapping is quite adhoc per component and should be defined in the
     * {@link FlowComponentQuirks} utility class.
     */
    private void createMethodCallChildrenStatements(CompilationUnit compilationUnit, InsertionPoint insertionPoint,
            JavaComponent parent, List<JavaComponent> methodChildren, Expression owner, String dataEntityRecordName) {
        List<JavaComponent> gridColumns = new ArrayList<>();
        List<JavaComponent> gridTreeColumns = new ArrayList<>();

        for (JavaComponent child : methodChildren) {
            if (FlowComponentQuirks.isGridTreeColumnDefinition(child)) {
                gridTreeColumns.add(child);
            }
            if (FlowComponentQuirks.isGridColumnDefinition(child)) {
                gridColumns.add(child);
            } else {
                // Get the method call expression for the child component (if
                // any
                List<MethodCallExpr> expression = FlowComponentQuirks.getMethodCallExprFromComponent(compilationUnit,
                        child, owner, dataEntityRecordName);
                for (MethodCallExpr methodCallExpr : expression) {
                    insertionPoint.add(new ExpressionStmt(methodCallExpr));
                }
            }
        }
        if (!gridColumns.isEmpty()) {
            // We use the items to filter columns that are not supported because of the type
            // of the data
            List<Map<String, Object>> itemsFromProperty = parent.getItemsFromProperty();
            List<String> supportedColumnsKeys = itemsFromProperty.stream().flatMap(map -> map.keySet().stream())
                    .collect(Collectors.toSet()).stream().collect(Collectors.toList());
            // We skip duplicate columns
            List<String> addedColumns = new ArrayList<>();
            MethodCallExpr expression = new MethodCallExpr(owner, "setColumns");
            for (JavaComponent gridColumn : gridColumns) {
                String columnName = gridColumn.props().get("path").toString();
                if (FlowComponentQuirks.isGridColumnDefinition(gridColumn) && supportedColumnsKeys.contains(columnName)
                        && !addedColumns.contains(columnName)) {
                    expression.addArgument(new StringLiteralExpr(columnName));
                    addedColumns.add(columnName);
                }
            }
            insertionPoint.add(new ExpressionStmt(expression));
        }
        if (!gridTreeColumns.isEmpty()) {
            MethodCallExpr expression = new MethodCallExpr(owner, "setHierarchyColumn");
            for (JavaComponent gridTreeColumn : gridTreeColumns) {
                expression.addArgument(new StringLiteralExpr(gridTreeColumn.props().get("path").toString()));
            }
            insertionPoint.add(new ExpressionStmt(expression));
        }
    }

    private void insertItemsPropToAddItem(CompilationUnit compilationUnit, JavaComponent javaComponent,
            InsertionPoint insertionPoint, Expression owner, Expression parent, String setterName, Object value) {
        if (javaComponent.tag().equals("MenuBar") && setterName.equals(ADD_ITEM_METHOD)) {
            JavaRewriterUtil.addImport(compilationUnit, "com.vaadin.flow.component.contextmenu.MenuItem");
            JavaRewriterUtil.addImport(compilationUnit, "com.vaadin.flow.component.contextmenu.SubMenu");
            FlowComponentQuirks.menuBarInsertItemsPropsToAddItem(javaComponent, insertionPoint, owner, parent,
                    setterName, value, false);
        } else {
            throw new IllegalArgumentException("Invalid or not supported items to be added separately for "
                    + javaComponent.tag() + " and setter " + setterName + " with value " + value.getClass().getName());
        }
    }

    /**
     * Adds a comment to the specified component in the source code.
     *
     * @param componentInfo
     *            the component to which the comment should be added
     * @param comment
     *            the comment to add
     * @param commentType
     *            the type of comment (LINE, BLOCK, JAVADOC)
     */
    public void addComment(ComponentInfo componentInfo, String comment, CommentType commentType) {
        Node targetNode = getNodeForAddingComment(componentInfo);

        if (targetNode != null) {
            Comment newComment = switch (commentType) {
            case LINE -> new LineComment(comment);
            case BLOCK -> new BlockComment(comment);
            };
            targetNode.setComment(newComment);
        } else {
            throw new IllegalArgumentException("Unable to determine the target node for the component.");
        }
    }

    /**
     * Helper method to get the appropriate Node from a ComponentInfo.
     *
     * @param componentInfo
     *            the component information
     * @return the corresponding Node, or null if not found
     */
    private Node getNodeForAddingComment(ComponentInfo componentInfo) {
        var createInfo = componentInfo.getCreateInfoOrThrow();
        if (createInfo.getLocalVariableDeclarator() != null) {
            return createInfo.getLocalVariableDeclarator().getParentNode().orElse(null);
        } else if (createInfo.getAssignmentExpression() != null) {
            return createInfo.getAssignmentExpression();
        } else if (componentInfo.getCreateInfoOrThrow().getFieldDeclaration() != null) {
            return componentInfo.getCreateInfoOrThrow().getFieldDeclaration();
        } else if (createInfo.getFieldDeclarationAndAssignment() != null) {
            return createInfo.getFieldDeclarationAndAssignment();
        } else if (componentInfo.routeConstructor() != null) {
            return componentInfo.routeConstructor();
        } else if (componentInfo.componentAttachInfoOptional().isPresent()
                && (componentInfo.componentAttachInfoOptional().get().getAttachCall() != null)) {
            return componentInfo.componentAttachInfoOptional().get().getAttachCall().getNode();
        }
        return null;
    }

    /**
     * Enum representing the type of comment to be added.
     */
    public enum CommentType {
        LINE, BLOCK
    }
}
