/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.vaadin.pro.licensechecker.dau;

import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.pro.licensechecker.Constants;
import com.vaadin.pro.licensechecker.LicenseException;
import com.vaadin.pro.licensechecker.LocalSubscriptionKey;
import com.vaadin.pro.licensechecker.Util;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonNull;
import elemental.json.JsonObject;

import static com.vaadin.pro.licensechecker.Util.TIMESTAMP_FORMAT;
import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * Daily Active Users publisher implementing Vaadin License Server DAU License
 * Model protocol.
 */
class LicenseServerPublisher implements Publisher {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(LicenseServerPublisher.class);
    static final String DEFAULT_PUBLISHING_URL = "https://tools.vaadin.com/vaadin-license-server";
    private final String publishingUrl;

    /**
     * Creates a new publisher syncing with Vaadin License Server.
     */
    public LicenseServerPublisher() {
        this(DEFAULT_PUBLISHING_URL);
    }

    /**
     * Creates a new publisher publishing data on the given URL.
     * <p>
     * </p>
     * The {@code publishingUrl} must be an absolute base URL pointing to a
     * Vaadin DAU License Model server. The base URL must not contain the
     * endpoint PATH {@literal /dau/check/{subscriptionKey}}.
     * <p>
     * </p>
     * An example of base URL is
     * {@literal https://staging.company.com:7777/dau-license-server}. Given
     * that base URL, the publisher will send HTTP POST requests to
     * {@literal https://staging.company.com:7777/dau-license-server/dau/check/{subscriptionKey}},
     * where {@literal subscriptionKey} is fetched at runtime by
     * {@link LocalSubscriptionKey}.
     *
     * @see LocalSubscriptionKey
     */
    public LicenseServerPublisher(String publishingUrl) {
        publishingUrl = Util.removeTrailingSlash(publishingUrl);
        publishingUrl += Constants.LICENSE_SERVER_DAU_PUBLISHING_URL;

        // Validate given URL
        this.publishingUrl = Util.validateURL(publishingUrl);
    }

    /**
     * Publishes collected tracked users for an application to an external
     * target, implementing Vaadin License Server DAU License Model protocol.
     *
     * @param applicationName
     *            the application name
     * @param trackedUsers
     *            collection of tracked Daily Active Users.
     * @return the enforcement rule to apply when tracking new users.
     * @throws LicenseException
     *             if a subscription key is not available or is not accepted by
     *             the License Server.
     * @throws PublishingException
     *             if an error occurs while publishing data.
     * @see LocalSubscriptionKey
     */
    @Override
    public DAUServerResponse publish(String applicationName,
            Collection<TrackedUser> trackedUsers, PublishingPhase phase) {
        LOGGER.debug("Publishing updates for application {}", applicationName);
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Tracked users: {}", trackedUsers);
        }
        JsonObject request = buildRequest(applicationName, trackedUsers, phase);
        try {
            JsonObject response = submit(request);
            DAUServerResponse dauServerResponse = parseResponse(response);
            LOGGER.debug("License Server enforcement policy: {}",
                    dauServerResponse.getEnforcementRule());
            return dauServerResponse;
        } catch (PublishingException | LicenseException ex) {
            throw ex;
        } catch (Exception ex) {
            throw new PublishingException(
                    "Publishing failed: " + ex.getMessage(), ex);
        }
    }

    private DAUServerResponse parseResponse(JsonObject response) {
        int status = (int) response.getNumber("status");
        response.remove("status");
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("License Server response: {}", response.toJson());
        }
        if (response.hasKey("enforce")
                && response.hasKey("rateLimitRemaining")) {
            try {
                Collection<TrackedUser> trackedUsers = parseTrackedUsers(
                        response);
                return new DAUServerResponse(new EnforcementRule(
                        response.getBoolean("enforce"),
                        Double.valueOf(response.getNumber("rateLimitRemaining"))
                                .intValue()),
                        trackedUsers);
            } catch (Exception ex) {
                throw new PublishingException(
                        "Invalid response from License Server: " + response,
                        ex);
            }
        } else if (response.hasKey("error") && response.hasKey("message")) {
            String errorMessage = "Error [" + response.getString("error") + "] "
                    + response.getString("message");
            if (response.hasKey("detail")) {
                errorMessage += ". " + response.getString("detail");
            }
            if (response.hasKey("path")) {
                errorMessage += ". (Path: " + response.getString("path") + ")";
            }

            if (status == 404) {
                throw new LicenseException(errorMessage);
            }
            throw new PublishingException(errorMessage);
        }
        throw new PublishingException(
                "Invalid response from License Server: " + response);
    }

    JsonObject buildRequest(String appIdentifier,
            Collection<TrackedUser> entries, PublishingPhase phase) {
        JsonObject request = Json.createObject();
        JsonArray users = Json.createArray();
        if (entries != null && !entries.isEmpty()) {
            entries.stream().flatMap(this::buildUsers)
                    .forEach(json -> users.set(users.length(), json));
        }

        request.put("appName", appIdentifier);
        request.put("timestamp", TIMESTAMP_FORMAT.format(Instant.now()));
        request.put("users", users);
        request.put("phase", phase.name());
        return request;
    }

    private Stream<JsonObject> buildUsers(TrackedUser user) {
        Stream<String> identities = user.getUserIdentityHashes().isEmpty()
                ? Stream.of((String) null)
                : user.getUserIdentityHashes().stream();
        return identities.map(identity -> {
            JsonObject json = Json.createObject();
            json.put("trackingHash", user.getTrackingHash());
            json.put("timestamp",
                    TIMESTAMP_FORMAT.format(user.getCreationTime()));
            if (identity != null) {
                json.put("userHash", identity);
            }
            return json;
        });
    }

    private List<TrackedUser> parseTrackedUsers(JsonObject response) {
        return parseTrackedUsersToMap(response).values().stream()
                .map(LicenseServerPublisher::pruneToSingleTrackedUser)
                .collect(Collectors.toList());
    }

    private static TrackedUser pruneToSingleTrackedUser(
            List<TrackedUser> trackedUsers) {
        TrackedUser user = new TrackedUser(
                trackedUsers.get(0).getTrackingHash(),
                trackedUsers.stream()
                        .max(Comparator.comparing(TrackedUser::getCreationTime))
                        .map(TrackedUser::getCreationTime)
                        .orElse(Instant.now().truncatedTo(ChronoUnit.SECONDS)));
        trackedUsers.stream().flatMap(
                trackedUser -> trackedUser.getUserIdentityHashes().stream())
                .forEach(user::linkUserIdentity);
        return user;
    }

    private Map<String, List<TrackedUser>> parseTrackedUsersToMap(
            JsonObject response) {
        return Optional.ofNullable(
                response.hasKey("users") ? response.getArray("users") : null)
                .filter(arr -> arr.length() > 0)
                .map(jsonArray -> IntStream.range(0, jsonArray.length())
                        .mapToObj(jsonArray::getObject))
                .orElseGet(Stream::empty).map(this::parseUser)
                .filter(Objects::nonNull)
                .collect(Collectors.groupingBy(TrackedUser::getTrackingHash));
    }

    private TrackedUser parseUser(JsonObject user) {
        if (!user.hasKey("trackingHash")) {
            return null;
        }
        try {
            TrackedUser trackedUser = new TrackedUser(
                    user.getString("trackingHash"), TIMESTAMP_FORMAT
                            .parse(user.getString("timestamp"), Instant::from));
            if (user.hasKey("userHash")
                    && !(user.get("userHash") instanceof JsonNull)) {
                trackedUser.linkUserIdentity(user.getString("userHash"));
            }
            return trackedUser;
        } catch (Exception e) {
            LOGGER.warn("Failed to parse user: {}", user);
            return null;
        }
    }

    private JsonObject submit(JsonObject request) throws IOException {
        String jsonString = request.toJson();
        LOGGER.trace("Submitting DAU request: {}", jsonString);

        HttpURLConnection http = getHttpURLConnection();

        try (OutputStream os = http.getOutputStream()) {
            os.write(jsonString.getBytes(UTF_8));
        }

        int status = http.getResponseCode();
        JsonObject response = null;
        if (status >= 200 && status < 300) {
            response = Util.parseJson(status, http.getInputStream());
        } else if (status >= 400 && status < 600) {
            response = Util.parseJson(status, http.getErrorStream());
        }
        if (response != null) {
            response.put("status", status);
            return response;
        }
        throw new IOException(
                "Unexpected response from License Server: " + status);
    }

    private HttpURLConnection getHttpURLConnection() throws IOException {
        URL url = new URL(publishingUrl);
        HttpURLConnection http = (HttpURLConnection) url.openConnection();

        http.setRequestMethod("POST");
        http.setRequestProperty("Content-Type", "application/json");
        http.setRequestProperty("Accept", "application/json");

        String subscriptionKey = LocalSubscriptionKey.getOrFail().getKey();
        http.setRequestProperty("Authorization", "DAU " + subscriptionKey);

        http.setConnectTimeout(5000);
        http.setReadTimeout(5000);
        http.setDoOutput(true);
        return http;
    }

}
