package io.takari.graph.tinkerpop;

//
// Tinkerpop graphs are naturally DFS and are topogically sorted as edges are inserted
//
import io.takari.graph.Dependency;
import io.takari.graph.DependencyGraph;
import io.takari.graph.render.D3GraphRenderer;
import io.takari.graph.render.DotGraphRenderer;
import io.takari.graph.render.GraphRenderer;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.util.List;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.tinkerpop.blueprints.Direction;
import com.tinkerpop.blueprints.Edge;
import com.tinkerpop.blueprints.Graph;
import com.tinkerpop.blueprints.Vertex;
import com.tinkerpop.blueprints.impls.tg.TinkerGraph;

public class TinkerPopDependencyGraph<V> implements DependencyGraph<V> {

  private static final String DATA = "@data@";
  private static final String ROOT = "@root@";
  private final Graph graph;

  public TinkerPopDependencyGraph(V root) {
    this.graph = new TinkerGraph();
    Vertex rootVertex = graph.addVertex(ROOT);
    rootVertex.setProperty(DATA, root);
  }

  @Override
  public V getRoot() {
    return graph.getVertex(ROOT).getProperty(DATA);
  }

  @Override
  public Iterable<V> getDependencyNodes() {
    return Iterables.transform(graph.getVertices(), new Function<Vertex, V>() {
      @Override
      public V apply(Vertex vertex) {
        return vertex.getProperty(DATA);
      }
    });
  }

  @Override
  public Dependency<V> addDependency(V source, V target) {
    Vertex sv = graph.getVertex(source);
    if (sv == null) {
      sv = graph.addVertex(source);
      sv.setProperty(DATA, source);
    }
    Vertex tv = graph.getVertex(target);
    if (tv == null) {
      tv = graph.addVertex(target);
      tv.setProperty(DATA, target);
    }
    Edge edge = graph.addEdge(null, sv, tv, "");
    Dependency<V> dependency = new Dependency<V>(source, target);
    edge.setProperty(DATA, dependency);
    return dependency;
  }

  @Override
  public Iterable<Dependency<V>> getDependencies() {
    return Iterables.transform(graph.getEdges(), new Function<Edge, Dependency<V>>() {
      @Override
      public Dependency<V> apply(Edge edge) {
        return edge.getProperty(DATA);
      }
    });
  }

  @Override
  public List<V> getDirectUpstream(V v) {
    Iterable<Vertex> upstream = graph.getVertex(v).getVertices(Direction.OUT);
    return Lists.newArrayList(Iterables.transform(upstream, new Function<Vertex, V>() {
      @Override
      public V apply(Vertex vertex) {
        return vertex.getProperty(DATA);
      }
    }));
  }

  @Override
  public List<V> getTransitiveUpstream(V v) {
    List<Vertex> upstream = Lists.newArrayList();
    Vertex vertex = graph.getVertex(v);
    collectVertices(vertex, Direction.OUT, upstream);
    return Lists.transform(upstream, new Function<Vertex, V>() {
      @Override
      public V apply(Vertex vertex) {
        return vertex.getProperty(DATA);
      }
    });
  }

  public List<V> getDirectDownstream(V v) {
    Iterable<Vertex> downstream = graph.getVertex(v).getVertices(Direction.IN);
    return Lists.newArrayList(Iterables.transform(downstream, new Function<Vertex, V>() {
      @Override
      public V apply(Vertex vertex) {
        return vertex.getProperty(DATA);
      }
    }));
  }

  public List<Vertex> getDownstreamTransitive(Vertex v) {
    List<Vertex> downstream = Lists.newArrayList();
    collectVertices(v, Direction.IN, downstream);
    return downstream;
  }

  private void collectVertices(Vertex vertex, Direction direction, List<Vertex> vertices) {
    for (Vertex v : vertex.getVertices(direction)) {
      vertices.add(v);
      collectVertices(v, direction, vertices);
    }
  }

  public static void main(String[] args) throws Exception {
    DependencyGraph<String> g = new TinkerPopDependencyGraph<String>("root");
    g.addDependency("root", "a");
    g.addDependency("root", "b");
    g.addDependency("root", "c");
    g.addDependency("root", "d");
    g.addDependency("a", "one");
    g.addDependency("a", "two");
    g.addDependency("b", "one");
    g.addDependency("b", "two");
    g.addDependency("c", "one");
    g.addDependency("c", "two");
    g.addDependency("d", "one");
    g.addDependency("d", "two");
    g.addDependency("d", "three");

    GraphRenderer<String> dot = new DotGraphRenderer<String>();
    dot.render(g, System.out);

    System.out.println();

    GraphRenderer<String> d3 = new D3GraphRenderer<String>();
    OutputStream os = new FileOutputStream(new File("/Users/jvanzyl/js/tesla/tesla-graph/src/main/webapp/maven.json"));
    d3.render(g, os);
  }
}
