package io.takari.project.generator;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.util.Map;

import org.apache.maven.model.Dependency;
import org.apache.maven.model.Model;
import org.apache.maven.model.Parent;
import org.apache.maven.model.io.xpp3.MavenXpp3Writer;

import com.github.mustachejava.DefaultMustacheFactory;
import com.github.mustachejava.Mustache;
import com.github.mustachejava.MustacheFactory;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
//
// generate complete project
// deploy generation
// vary maven parameters and build
//

import io.takari.graph.dot.Edge;
import io.takari.graph.dot.Graph;
import io.takari.graph.dot.Node;
import io.takari.resolution.Artifact;

public abstract class AbstractProjectGenerator implements ProjectGenerator {

  private final File dot;
  private final JavaSourceGenerator java;
  
  public AbstractProjectGenerator(File dot) {
    this.dot = dot;
    this.java = new JavaSourceGenerator();
  }
  
  @Override
  public void generate(File outputDirectory) throws Exception {

    Graph graph = parse(dot);
    String parentArtifactId = graph.attributes().get("artifactId");
    String groupId = graph.attributes().get("groupId");
    String version = graph.attributes().get("version");
    String versionRange = graph.attributes().get("versionRange");

    Preconditions.checkArgument(groupId != null, "You must specify a groupId attribute in the graph.");
    Preconditions.checkArgument(version != null, "You must specify a version attribute in the graph.");

    graph.attributes().put("home", System.getProperty("user.home"));
    
    Map<String, Model> models = Maps.newLinkedHashMap();
    for (Node n : graph.getNodes()) {
      String artifactId = n.getId();
      if (!is3rdParty(artifactId)) {
        Model model = model(parentArtifactId, versionRange, groupId, artifactId, version);
        models.put(artifactId, model);
      }
    }

    if (graph.getEdges() != null) {
      for (Edge e : graph.getEdges()) {
        String source = e.getStart().getId();
        String target = e.getEnd().getId();
        Model model = models.get(source);
        Dependency d = new Dependency();
        //
        // The target might be a 3rd party dependency
        //
        if (is3rdParty(target)) {
          Artifact a = new Artifact(target.replaceAll("^\"|\"$", ""));
          d.setGroupId(a.getGroupId());
          d.setArtifactId(a.getArtifactId());
          d.setVersion(a.getVersion());
          if (a.getClassifier() != null) {
            d.setClassifier(a.getClassifier());
          }
        } else {
          d.setGroupId(groupId);
          d.setArtifactId(target);
          d.setVersion(versionRange);
        }
        model.addDependency(d);
      }
    }

    //
    // Write out modules
    //
    for (Model m : models.values()) {
      File projectOutputDirectory = new File(outputDirectory, m.getArtifactId());
      File pom = new File(projectOutputDirectory, "pom.xml");
      if (!pom.getParentFile().exists()) {
        pom.getParentFile().mkdirs();
      }
      System.out.println("Generating " + m);
      write(m, new OutputStreamWriter(new FileOutputStream(pom)));
      String packageName = groupId;
      String className = "App";
      File source = new File(projectOutputDirectory, "src/main/java");
      java.generate(packageName, className, source);
    }

    //
    // Write root pom
    //
    Model model = model(groupId, parentArtifactId, version);
    model.setPackaging("pom");
    for (Model m : models.values()) {
      model.addModule(m.getArtifactId());
    }
    File pom = new File(outputDirectory, "pom.xml");
    if (!pom.getParentFile().exists()) {
      pom.getParentFile().mkdirs();
    }
    generate(model, graph.attributes(), new OutputStreamWriter(new FileOutputStream(pom)));
    System.out.println("Generating " + model);
    //
    // Write out additional configurations
    //
    fromTemplate("workspace.xml", graph.attributes(), outputDirectory);
    fromTemplate("workspace-user.xml", graph.attributes(), outputDirectory);
    //fromTemplate("settings.xml", graph.attributes(), outputDirectory);
    System.out.println("Complete.");
  }

  public abstract Graph parse(File dot) throws Exception;

  private static void write(Model model, Writer writer) throws Exception {
    MavenXpp3Writer modelWriter = new MavenXpp3Writer();
    modelWriter.write(writer, model);
  }

  private Model model(String parentArtifactId, String parentVersion, String groupId, String artifactId, String version) {
    Model model = model(groupId, artifactId, version);
    Parent parent = new Parent();
    parent.setGroupId(groupId);
    parent.setArtifactId(parentArtifactId);
    parent.setVersion(parentVersion);
    model.setParent(parent);
    model.setPackaging("${projectLifecycle}");
    return model;
  }

  private Model model(String groupId, String artifactId, String version) {
    Model model = new Model();
    model.setModelVersion("4.0.0");
    model.setGroupId(groupId);
    model.setArtifactId(artifactId);
    model.setVersion(version);
    return model;
  }

  public void generate(Model model, Object context, Writer writer) throws IOException {
    MustacheFactory mf = new DefaultMustacheFactory();
    try (InputStream is = getClass().getClassLoader().getResourceAsStream("pom-parent.mustache")) {
      Reader reader = new InputStreamReader(is);
      Mustache mustache = mf.compile(reader, "project");
      mustache.execute(writer, new Object[] {model, context}).flush();
    }
  }

  public void fromTemplate(String template, Object context, File outputDirectory) throws IOException {
    MustacheFactory mf = new DefaultMustacheFactory();
    File target = new File(outputDirectory, template);
    try (InputStream is = getClass().getClassLoader().getResourceAsStream(template); Writer writer = new OutputStreamWriter(new FileOutputStream(target))) {
      Reader reader = new InputStreamReader(is);
      Mustache mustache = mf.compile(reader, "project");
      mustache.execute(writer, context).flush();
    }
  }

  boolean is3rdParty(String nodeId) {
    return (nodeId.length() - nodeId.replace(":", "").length()) >= 2;
  }
}
