package io.takari.resolution;

import java.io.Serializable;

public class Artifact implements Serializable {

  private static final long serialVersionUID = 1L;

  private final String groupId;
  private final String artifactId;
  private final String version;
  private final String classifier;
  private final String extension;
  private final String coordinate;

  /**
   * Creates a new artifact with the specified coordinates and properties. If not specified in the artifact
   * coordinates, the artifact's extension defaults to {@code jar} and classifier to null.
   * 
   * @param coords The artifact coordinates in the format {@code <groupId>:<artifactId>[:<extension>[:<classifier>]]:<version>}, must not be {@code null}.
   * @param properties The artifact properties, may be {@code null}.
   */
  public Artifact(String coords) {
    String[] parts = coords.split(":"); // does not use a regex with a single character so it's reasonably fast
    if (parts.length < 3) {
      throw new IllegalArgumentException("Bad artifact coordinates " + coords + ", expected format is <groupId>:<artifactId>[:<extension>[:<classifier>]]:<version>");
    }
    this.groupId = parts[0];
    this.artifactId = parts[1];
    switch (parts.length) {
    case 3:
      this.extension = "jar";
      this.classifier = null;
      this.version = parts[2];
      break;
    case 4:
      this.extension = parts[2];
      this.classifier = null;
      this.version = parts[3];
      break;
    case 5:
      this.extension = parts[2].isEmpty() ? "jar" : parts[2];
      this.classifier = parts[3];
      this.version = parts[4];
      break;
    default:
      throw new IllegalArgumentException("Bad artifact coordinates " + coords + ", expected format is <groupId>:<artifactId>[:<extension>[:<classifier>]]:<version>");
    }
    this.coordinate = coords;
  }

  public Artifact(String groupId, String artifactId, String classifier, String extension, String version) {
    this.groupId = groupId;
    this.artifactId = artifactId;
    this.classifier = classifier;
    this.extension = extension;
    this.version = version;
    this.coordinate = coordinate();
  }

  public String getGroupId() {
    return groupId;
  }

  public String getArtifactId() {
    return artifactId;
  }

  public String getVersion() {
    return version;
  }

  public String getClassifier() {
    return classifier;
  }

  public String getExtension() {
    return extension;
  }

  public String getCoordinate() {
    return coordinate;
  }

  public String coordinate() {
    //
    // <groupId>:<artifactId>[:<extension>[:<classifier>]]:<version>
    //    
    StringBuffer path = new StringBuffer() //
        .append(groupId) //
        .append(':') //
        .append(artifactId) //
        .append(':');

    if (extension != null) {
      path.append(extension) //
          .append(':');
      if (classifier != null) {
        path.append(classifier) //
            .append(':');
      }
    }
    path.append(version);
    return path.toString();
  }

  @Override
  public String toString() {
    return coordinate;
  }

}
